package ru.yandex.direct.chassis.util.startrek

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component
import ru.yandex.bolts.collection.Cf
import ru.yandex.bolts.collection.Option
import ru.yandex.direct.chassis.entity.startrek.affectedApps
import ru.yandex.direct.chassis.entity.startrek.testedApps
import ru.yandex.direct.chassis.util.Utils
import ru.yandex.direct.chassis.util.cf
import ru.yandex.direct.chassis.util.nonCf
import ru.yandex.direct.tracing.Trace
import ru.yandex.misc.io.http.UriBuilder
import ru.yandex.startrek.client.AuthenticatingStartrekClient
import ru.yandex.startrek.client.Session
import ru.yandex.startrek.client.model.ChecklistItem
import ru.yandex.startrek.client.model.Comment
import ru.yandex.startrek.client.model.CommentUpdate
import ru.yandex.startrek.client.model.Issue
import ru.yandex.startrek.client.model.IssueCreate
import ru.yandex.startrek.client.model.IssueRef
import ru.yandex.startrek.client.model.IssueUpdate
import ru.yandex.startrek.client.model.SearchRequest
import ru.yandex.startrek.client.model.UserRef
import java.net.URLEncoder
import java.nio.charset.StandardCharsets
import java.time.LocalDate

@Component
class StartrekHelper(
    @Value("\${startrek.startrek_url}") val startrekUrl: String,
    val session: Session,
) {
    private val logger = LoggerFactory.getLogger(StartrekHelper::class.java)
    private val tries = 5

    fun findIssues(query: String): List<Issue> {
        return session.issues().find(query).toList().nonCf()
    }

    fun findIssuesByKeys(keys: Collection<String>): List<Issue> {
        if (keys.isEmpty()) {
            return listOf()
        }
        val searchRequest = SearchRequest.builder()
            .keys(*keys.toTypedArray())
            .build()
        val issues = session.issues().find(searchRequest).toList().nonCf()

        val issueOrder: Map<String, Int> = keys.withIndex()
            .associate { (index, key) -> key.uppercase() to index }
        return issues
            .sortedBy { issueOrder[it.key.uppercase()] }
    }

    fun findIssueByKey(key: String): Issue? {
        return findIssuesByKeys(listOf(key)).firstOrNull()
    }

    /**
     * Добавить комментарий к тикету в Startrek
     * @param comment текст комментария, к нему будет добавлена отбивка про текущий service.method
     * @param notify нужно ли отправлять нотификации
     * @param ifNoTag слабая попытка CAS - перед добавлением комментария проверить есть ли тег
     * @param summonee кого призвать в комментарий
     * @param footer добавлять ли к сообщению подпись с названием джобы
     */
    fun addComment(
        issue: IssueRef,
        comment: String,
        notify: Boolean = true,
        ifNoTag: String? = null,
        summonee: UserRef? = null,
        summonees: List<UserRef>? = null,
        footer: Boolean = true,
    ) {
        if (ifNoTag != null && hasTag(issue, ifNoTag)) {
            logger.info("Issue ${issue.key} has tag $ifNoTag, skip comment addition")
            return
        }

        val commentWithFooter = if (footer) {
            comment + "\n" + createFooter()
        } else {
            comment
        }

        val update = issueUpdate {
            when {
                summonees != null -> comment(commentWithFooter, *summonees.toTypedArray())
                summonee != null -> comment(commentWithFooter, summonee)
                else -> comment(commentWithFooter, Cf.list())
            }

            if (ifNoTag != null) {
                tags(listOf(ifNoTag).cf, listOf<String>().cf)
            }
        }

        session.issues().update(issue, update, notify, notify)
    }

    fun updateCommentText(issue: Issue, comment: Comment, commentText: String, notify: Boolean = false) {
        val commentUpdate = CommentUpdate.builder().comment(commentText).build()
        session.comments().update(issue, comment, commentUpdate, notify, notify)
    }

    fun hasTag(issue: IssueRef, ifNoTag: String): Boolean {
        val freshCopy = session.issues().get(issue.id)
        return freshCopy.tags.nonCf().contains(ifNoTag)
    }

    /**
     * Добавить к тикету тэг
     */
    fun addTag(issue: Issue, tag: String) =
        Utils.retry(tries) {
            session.issues().update(issue, issueUpdate {
                tags(Cf.list(tag), Cf.list())
            })
        }

    /**
     * Удалить тег тикета
     */
    fun removeTags(issue: Issue, tags: List<String>) =
        Utils.retry(tries) {
            session.issues().update(issue, issueUpdate {
                tags(Cf.list(), Cf.toList(tags))
            })
        }

    /**
     * Добавить к тикету чеклист
     * @param ifNoTag чеклист не будет добавлен, если на тикете есть этот тег
     */
    fun addChecklist(issue: Issue, items: List<String>, ifNoTag: String? = null) {
        if (ifNoTag != null && hasTag(issue, ifNoTag)) {
            logger.info("Issue ${issue.key} have tag $ifNoTag, skip checklist addition")
            return
        }

        val checklistItems = items.map {
            ChecklistItem.Builder().text(it).checked(false).build()
        }

        session.issues().addChecklistItems(issue, checklistItems)
        if (ifNoTag != null) {
            addTag(issue, ifNoTag)
        }
    }

    /**
     * Обновить элемент [itemId] чеклиста значениями [update]
     */
    fun updateChecklist(issue: Issue, itemId: String, update: ChecklistItemUpdate) {
        val client = session as AuthenticatingStartrekClient
        client.doPatch(
            UriBuilder.cons(client.endpoint)
                .appendPath("issues").appendPath(issue.id)
                .appendPath("checklistItems").appendPath(itemId)
                .build(),
            update,
            client.factory().cons(Issue::class.java),
        )
    }

    /**
     * Посчитать число тикетов по фильтру
     */
    fun issuesCount(filter: String) =
        session.issues().find(filter).count()

    /**
     * Добавить приложение [app] в поле `Affected apps`. Не делает валидацию для [app]
     */
    fun appendAffectedApp(issue: Issue, app: String) {
        session.issues().update(
            issue,
            issueUpdate { set(StartrekField.AFFECTED_APPS, issue.affectedApps + app) },
            false,
            false
        )
    }

    /**
     * Добавить приложение [app] в поле `Tested apps`. Не делает валидацию для [app]
     */
    fun appendTestedApp(issue: Issue, app: String) {
        session.issues().update(
            issue,
            issueUpdate { set(StartrekField.TESTED_APPS, issue.testedApps + app) },
            false,
            false
        )
    }

    /**
     * Обновить статус тикета. `status` должен быть в camelCase
     */
    fun updateStatus(issue: Issue, status: String, issueUpdate: IssueUpdate? = null): Boolean {
        return try {
            when (issueUpdate) {
                null -> issue.executeTransition(status)
                else -> issue.executeTransition(status, issueUpdate)
            }
            true
        } catch (e: RuntimeException) {
            logger.warn("Failed to update status $status for $issue", e)
            false
        }
    }

    /**
     * Пытается закрыть тикет (перевести в `closed`).
     * Без [force] может закрыть только тикет, у которого есть переход в `closed`.
     * Иначе, попробует перевести тикет в `open`, а затем в `closed`.
     *
     * @param force Если `true`, то при отсутствии перехода в `closed` попробует сначала перевести в `open`
     * @return `false`, если не получилось сменить статус, иначе `true`
     */
    fun closeIssue(issue: Issue, force: Boolean = false): Boolean {
        if (issue.status.key == StartrekStatusKey.CLOSED) {
            return true
        }

        val closeTransition = issue.transitions.nonCf()
            .find { it.to.key == StartrekStatusKey.CLOSED }
        if (closeTransition != null) {
            return updateStatus(issue, closeTransition.id, IssueUpdate.resolution(StartrekResolution.FIXED).build())
        }

        if (force) {
            val openTransition = issue.transitions.nonCf()
                .find { it.to.key == StartrekStatusKey.OPEN }
                ?: return false
            if (!updateStatus(issue, openTransition.id)) {
                return false
            }
            return closeIssue(issue, force = false)
        }

        return false
    }

    /**
     * Создает подпись с названием джобы и ссылкой на её лог
     */
    fun createFooter(): String {
        val trace = Trace.current()
        val from = LocalDate.now().atStartOfDay()
        val to = LocalDate.now().plusDays(1).atStartOfDay()

        val logViewerLink = Utils.logViewerLink(from, to, trace.service, trace.traceId)
        return """
            |----
            |(($logViewerLink ${trace.service}.${trace.method}))
        """.trimMargin()
    }

    companion object {
        fun issueUpdate(init: IssueUpdate.Builder.() -> Unit): IssueUpdate {
            val builder = IssueUpdate.builder()
            builder.init()
            return builder.build()
        }

        // получение значения опционального поля через два опшинала
        fun <T> Issue.field(name: String): T? =
            this.getO<Option<T>>(name).orNull?.orNull

        fun filterLink(query: String, startrekUrl: String) =
            "${startrekUrl}/filters/filter?query=${URLEncoder.encode(query, StandardCharsets.UTF_8)}"

        fun IssueCreate.Builder.components(components: List<String>): IssueCreate.Builder {
            set("components", components)
            return this
        }
    }
}
