package ru.yandex.direct.chassis.util.ydb;

import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.Value;
import one.util.streamex.StreamEx;

import ru.yandex.bolts.collection.Option;   // IGNORE-BAD-STYLE DIRECT-117880
import ru.yandex.startrek.client.model.Field;
import ru.yandex.startrek.client.model.Issue;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Класс для описания связей "поле в трекере - поле модели - столбец в YDB - столбец в таблице".
 *
 * @param <T> тип поля в трекере
 * @param <M> тип модели в которой хранятся поля
 * @param <Y> тип, выгружаемый в YDB
 */
public class Mapping<T, M, Y> {
    private final String ydbName;
    private final PrimitiveType ydbType;
    private final Function<Y, Value<PrimitiveType>> ydbConverter;
    private final Function<M, Y> getterForYdb;

    private final String trackerName;
    private final Field.Schema trackerType;
    private final BiConsumer<M, T> setterFromTracker;

    private Mapping(Builder<T, M, Y> builder) {
        this.ydbName = builder.ydbName;
        this.ydbType = builder.ydbType;
        this.ydbConverter = builder.ydbConverter;
        this.getterForYdb = builder.getterForYdb;
        this.trackerName = builder.trackerName;
        this.trackerType = builder.trackerType;
        this.setterFromTracker = builder.setterFromTracker;
    }

    public static class Builder<T2, M2, Y2> {
        private String ydbName;
        private PrimitiveType ydbType;
        private Function<Y2, Value<PrimitiveType>> ydbConverter;
        private Function<M2, Y2> getterForYdb;

        private String trackerName;
        private Field.Schema trackerType;
        private BiConsumer<M2, T2> setterFromTracker;

        private Builder(BiConsumer<M2, T2> setter, Function<M2, Y2> getter) {
            getterForYdb = getter;
            setterFromTracker = setter;
        }

        public Builder<T2, M2, Y2> withYdbSpec(String columnName, PrimitiveType columnType,
                                               Function<Y2, Value<PrimitiveType>> converter) {
            ydbName = columnName;
            ydbType = columnType;
            ydbConverter = converter;
            return this;
        }

        public Builder<T2, M2, Y2> withTrackerSpec(String fieldName, Field.Schema fieldType) {
            trackerName = fieldName;
            trackerType = fieldType;
            return this;
        }

        public Mapping<T2, M2, Y2> build() {
            return new Mapping<>(this);
        }
    }

    /**
     * Билдер для колонки, которая должна быть прочитана из трекера и записана в YDB
     *
     * @param modelGetterToYdbColumn      метод чтения поля из модели (для записи значения в YDB)
     * @param modelSetterFromTrackerField метод для записи поля в модель (прочитанного из трекера)
     */
    public static <T2, M2, Y2> Builder<T2, M2, Y2> readWrite(Function<M2, Y2> modelGetterToYdbColumn,
                                                             BiConsumer<M2, T2> modelSetterFromTrackerField) {
        return new Builder<>(modelSetterFromTrackerField, modelGetterToYdbColumn);
    }

    /**
     * Билдер для вычислимой колонки (не читается напрямую из трекера), которую нужно записать в YDB
     *
     * @param modelGetterToYdbColumn метод чтения поля из модели (для записи значения в YDB)
     */
    public static <M2, Y2> Builder<Void, M2, Y2> read(Function<M2, Y2> modelGetterToYdbColumn) {
        return new Builder<>(null, modelGetterToYdbColumn);
    }

    /**
     * Билдер для колонки читаемой из трекера, без записи в YDB.
     *
     * @param modelSetterFromTrackerField метод для записи поля в модель (прочитанного из трекера)
     */
    public static <T2, M2> Builder<T2, M2, Void> write(BiConsumer<M2, T2> modelSetterFromTrackerField) {
        return new Builder<>(modelSetterFromTrackerField, null);
    }

    public void fillFromIssue(M incident, Issue issue) {
        T value;
        if ("key".equals(trackerName)) {
            // ключ тикета не попадает в values, поэтому вот такой костыль
            //noinspection unchecked
            value = (T) issue.getKey();
        } else if (isOptionalField(trackerType)) {
            // некоторые опциональные поля упакованным в Option, некоторых вообще отсутствуют в issue.values
            value = issue.<Option<T>>getO(trackerName).getOrElse(Option.empty()).getOrNull();
        } else if (isArrayField(trackerType)) {
            //noinspection unchecked (из-за пустого списка в else)
            value = issue.<T>getO(trackerName).getOrElse((T) List.of());
        } else {
            value = issue.get(trackerName);
        }
        setterFromTracker.accept(incident, value);
    }

    @Nullable
    public String getTrackerName() {
        return trackerName;
    }

    @Nullable
    public Field.Schema getTrackerType() {
        return trackerType;
    }

    @Nullable
    public String getYdbName() {
        return ydbName;
    }

    @Nullable
    public PrimitiveType getYdbType() {
        return ydbType;
    }

    @Nullable
    public Value<PrimitiveType> getValueForYdb(M model) {
        checkNotNull(getterForYdb);
        Y value = getterForYdb.apply(model);
        if (value == null) {
            return null;
        }
        return ydbConverter.apply(value);
    }

     boolean canBeSavedToYdb() {
        return ydbType != null
                && getterForYdb != null
                && ydbConverter != null;
     }

    public static <M> Map<String, Field.Schema> getCustomFields(List<Mapping<?, M, ?>> mappings) {
        return StreamEx.of(mappings)
                .mapToEntry(Mapping::getTrackerName, Mapping::getTrackerType)
                .nonNullKeys()
                .toImmutableMap();
    }

    public static <M> TableDescription.Builder getTableDescriptionBuilder(List<Mapping<?, M, ?>> mappings) {
        TableDescription.Builder builder = TableDescription.newBuilder();
        StreamEx.of(mappings)
                .mapToEntry(Mapping::getYdbName, Mapping::getYdbType)
                .nonNullKeys()
                .forKeyValue(builder::addNullableColumn);
        return builder;
    }

    private static boolean isOptionalField(Field.Schema schema) {
        return schema instanceof Field.Schema.Scalar && !((Field.Schema.Scalar) schema).isRequired();
    }

    private static boolean isArrayField(Field.Schema schema) {
        return schema instanceof Field.Schema.Array;
    }
}
