package ru.yandex.direct.chassis.util.ydb;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static ru.yandex.direct.ytwrapper.model.YtDynamicOperator.getWithTimeout;

@ParametersAreNonnullByDefault
public class YdbClient {
    private static final Logger logger = LoggerFactory.getLogger(YdbClient.class);

    public static final ExecuteDataQuerySettings KEEP_IN_CACHE = new ExecuteDataQuerySettings().keepInQueryCache();
    public static final String V1_SYNTAX = "--!syntax_v1\n";

    private final SessionRetryContext sessionRetryContext;
    private final String database;
    private final Duration timeout;

    public YdbClient(String endpoint, String database, AuthProvider authProvider) {
        this(endpoint, database, authProvider, Duration.ofSeconds(60));
    }

    public YdbClient(String endpoint, String database, AuthProvider authProvider, Duration timeout) {
        var transport = GrpcTransport.forEndpoint(endpoint, database)
                .withAuthProvider(authProvider)
                .build();
        var tableRpc = GrpcTableRpc.useTransport(transport);
        var tableClient = TableClient.newClient(tableRpc)
                .sessionPoolSize(2, 15)
                .build();

        this.sessionRetryContext = SessionRetryContext.create(tableClient)
                .maxRetries(5)
                .retryNotFound(true)
                .build();
        this.database = database;
        this.timeout = timeout;
    }

    public void createTable(String tableName, TableDescription tableDescription) {
        String path = database + "/" + tableName;
        logger.debug("create table {}", path);
        supplyStatus(session -> session.createTable(path, tableDescription))
                .expect("Error creating table " + tableName);
    }

    public Status supplyStatus(Function<Session, CompletableFuture<Status>> fn) {
        CompletableFuture<Status> future = sessionRetryContext.supplyStatus(fn);
        return getWithTimeout(future, timeout, "supplyStatus timeout out");
    }

    public <T> Result<T> supplyResult(Function<Session, CompletableFuture<Result<T>>> fn) {
        CompletableFuture<Result<T>> future = sessionRetryContext.supplyResult(fn);
        return getWithTimeout(future, timeout, "supplyResult timeout out");
    }
}
