package ru.yandex.direct.chassis.util.ydb;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.values.OptionalType;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.Value;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.startrek.client.model.Issue;

import static com.yandex.ydb.table.transaction.TxControl.serializableRw;
import static ru.yandex.direct.chassis.util.ydb.YdbClient.KEEP_IN_CACHE;

/**
 * Общий код для сохранения из модели (описанной маппингом) в таблицу YDB
 *
 * @param <M> тип модели в которой хранятся поля
 */
public class YdbTableWriter<M> {
    private static final Logger logger = LoggerFactory.getLogger(YdbTableWriter.class);

    private final YdbClient ydb;
    private final String tableName;
    private final List<Mapping<?, M, ?>> mappings;
    private final Supplier<M> modelSupplier;

    public YdbTableWriter(YdbClient ydb, String tableName, Supplier<M> modelSupplier, List<Mapping<?, M, ?>> mappings) {
        this.ydb = ydb;
        this.tableName = tableName;
        this.modelSupplier = modelSupplier;
        this.mappings = mappings;
    }

    private static String getVarName(String column) {
        return "$" + column;
    }

    /**
     * Выполняет UPSERT для всех колонок, которые имеют представление в YDB.
     */
    public void saveToYdb(M model) {
        var data = new YdbData();
        data.fillFromModel(model, Set.of());

        if (data.params.isEmpty()) {
            return;
        }
        logger.info("saving {} to YDB {} table", model, tableName);

        String values = data.columns.stream()
                .map(YdbTableWriter::getVarName)
                .collect(Collectors.joining(","));
        String query = data.getDeclares() + "\n"
                + "UPSERT INTO `" + tableName + "`"
                + "(" + String.join(",", data.columns) + ")"
                + "VALUES(" + values + ")";

        ydb.supplyResult(session -> session.executeDataQuery(query, serializableRw(), data.params, KEEP_IN_CACHE))
                .expect("error replacing row");
    }

    /**
     * Делает update по всем представимым в YDB полям модели, кроме keyColumns (они идут в WHERE) и skipColumns.
     */
    public void updateInYdb(M model, Set<String> keyColumns, Set<String> skipColumns) {
        var data = new YdbData();
        data.fillFromModel(model, skipColumns);

        if (data.columns.isEmpty()) {
            return;
        }
        logger.info("update {} in YDB {} table by {}", model, tableName, keyColumns);

        String values = data.columns.stream()
                .filter(c -> !keyColumns.contains(c))
                .map(column -> column + " = " + getVarName(column))
                .collect(Collectors.joining(", "));
        String where = keyColumns.stream()
                .map(column -> column + " = " + getVarName(column))
                .collect(Collectors.joining(" AND "));

        String query = data.getDeclares() + "\n"
                + "UPDATE `" + tableName + "` SET "
                + values
                + " WHERE " + where;

        ydb.supplyResult(session -> session.executeDataQuery(query, serializableRw(), data.params, KEEP_IN_CACHE))
                .expect("error updating rows");
    }

    public M parseIssue(Issue issue) {
        logger.trace("parse issue {}", issue.getKey());
        try {
            M incident = modelSupplier.get();

            StreamEx.of(mappings)
                    .filter(m -> m.getTrackerName() != null)
                    .forEach(m -> m.fillFromIssue(incident, issue));

            return incident;
        } catch (RuntimeException e) {
            logger.error("Failed to parse " + issue.getKey(), e);
            throw e;
        }
    }

    private class YdbData {
        private final Params params;
        private final List<String> declares;
        private final List<String> columns;

        private YdbData() {
            params = Params.create();
            declares = new ArrayList<>();
            columns = new ArrayList<>();
        }

        void fillFromModel(M model, Set<String> skipColumns) {
            for (Mapping<?, M, ?> mapping : mappings) {
                if (!mapping.canBeSavedToYdb()) {
                    continue;
                }

                OptionalType optionalType = mapping.getYdbType().makeOptional();
                OptionalValue optionalValue;
                Value<PrimitiveType> value = mapping.getValueForYdb(model);
                if (value == null) {
                    optionalValue = optionalType.emptyValue();
                } else {
                    optionalValue = optionalType.newValue(value);
                }

                String name = mapping.getYdbName();
                String varName = getVarName(name);
                if (skipColumns.contains(name)) {
                    continue;
                }

                declares.add(String.format("DECLARE %s AS %s;", varName, optionalType));
                columns.add(name);
                params.put(varName, optionalValue);
            }
        }

        String getDeclares() {
            return String.join("\n", declares);
        }
    }
}
