package ru.yandex.direct.currencyimporter;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Stream;

import javax.annotation.Generated;
import javax.lang.model.element.Modifier;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.currency.ConstantDescription;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Rate;

import static com.google.common.base.CaseFormat.UPPER_CAMEL;
import static com.google.common.base.CaseFormat.UPPER_UNDERSCORE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Загружает, валидирует, трансформирует и сохраняет Java файлы с информацией о валютах.
 * Генерируется интерфейс, описывающий валютные констаны перечисленные в загруженном JSON (+ код валюты),
 * Также генерируются классы валют, реализующие этот интерфейс, значения констант берутся из JSON.
 * В отдельный класс загружаются курсы валют
 * <p>
 * При появлении новых констант, возможно потребуются уточнение их типа в {@link #getTypeByField(String)}
 * и {@link #makeClassField(String, JsonNode)}
 * </ul>
 */
public class CurrencyImporter {
    private static final TypeReference<HashMap<String, String>> TYPE_REF_FIELDS =
            new TypeReference<HashMap<String, String>>() {
            };
    private static final String CODE_FIELD = "CURRENCY_CODE";
    private static final String INDENT = "    ";

    private static final String CURRENCIES_ROOT_PROPERTY = "currencies";
    private static final String FIELDS_ROOT_PROPERTY = "fields";

    private static final ImmutableSet<String> MANDATORY_CURRENCY_CODES =
            ImmutableSet.of("RUB", "YND_FIXED", "USD", "EUR");
    private static final int CURRENCY_COUNT_LIMIT = 300;

    private final URL currenciesRemoteJson;
    private final URL ratesRemoteJson;
    private String currenciesJsonBody;
    private String ratesJsonBody;
    private HashMap<String, String> currencyFields;
    private ArrayList<JsonNode> currencies;
    private TreeMap<String, TreeMap<String, Pair<String, String>>> rates;
    private final String classNamePrefix;
    private final File destRoot;
    private final String interfacePackage;
    private final String currenciesPackage;
    private final AnnotationSpec generatedCurrenciesAnnotation;
    private final AnnotationSpec generatedRatesAnnotation;

    public CurrencyImporter(URL currenciesRemoteJson, URL ratesRemoteJson, File destRoot,
                            String currenciesPackage, String interfacePackage,
                            String classNamePrefix) {
        this.currenciesRemoteJson = currenciesRemoteJson;
        this.ratesRemoteJson = ratesRemoteJson;
        this.classNamePrefix = classNamePrefix;
        this.destRoot = destRoot;
        this.interfacePackage = interfacePackage;
        this.currenciesPackage = currenciesPackage;
        this.generatedCurrenciesAnnotation = getGeneratedAnnotation(currenciesRemoteJson);
        this.generatedRatesAnnotation = getGeneratedAnnotation(ratesRemoteJson);
    }

    private AnnotationSpec getGeneratedAnnotation(URL remoteJson) {
        return AnnotationSpec.builder(Generated.class)
                .addMember("value", "$S", CurrencyImporter.class.getCanonicalName())
                .addMember("date", "$S", LocalDate.now().toString())
                .addMember("comments", "$S", "generated from " + remoteJson.toString())
                .build();
    }

    private void fetchRemoteJsons() {
        try (InputStream is = currenciesRemoteJson.openStream()) {
            currenciesJsonBody = IOUtils.toString(is, StandardCharsets.UTF_8);
        } catch (IOException ex) {
            throw new CurrencyImportException("can't read currencies remote JSON", ex);
        }

        try (InputStream is = ratesRemoteJson.openStream()) {
            ratesJsonBody = IOUtils.toString(is, StandardCharsets.UTF_8);
        } catch (IOException ex) {
            throw new CurrencyImportException("can't read rates remote JSON", ex);
        }
    }

    /**
     * Валидирует JSON.
     * Проверка считается успешной, если:
     * <ul>
     * <li>удается преобразовать JSON во внутреннее представление (см. {@link #deserializeCurrenciesJson()} )</li>
     * <li>преобразованный Collection имеет размер от 0 до {@link #CURRENCY_COUNT_LIMIT}</li>
     * <li>в коллекции присутствуют валюты с кодами из  {@link #MANDATORY_CURRENCY_CODES}</li>
     * <li>загружен курс для каждой валюты кроме YND_FIXED</li>
     * </ul>
     *
     * @throws CurrencyImportException если JSON не валиден
     */
    private void validateData() {
        if (currencies.isEmpty() || currencies.size() > CURRENCY_COUNT_LIMIT) {
            String errorMsg = String.format("currency map size must be between 0 and %d(actual size: %d)",
                    CURRENCY_COUNT_LIMIT, currencies.size());
            throw new CurrencyImportException(errorMsg);
        }
        if (currencies.size() != rates.size() + 1) {
            String errorMsg = "rate map size must be equal currency map size + 1";
            throw new CurrencyImportException(errorMsg);
        }
        Set<String> actualCurrencyCodes = ImmutableSet.copyOf(mapList(currencies, c -> c.get(CODE_FIELD).textValue()));
        Set<String> missedMandatoryFields = Sets.difference(MANDATORY_CURRENCY_CODES, actualCurrencyCodes);
        if (!missedMandatoryFields.isEmpty()) {
            String errorMsg = String.format("%s codes are mandatory (absent codes: %s)",
                    MANDATORY_CURRENCY_CODES, missedMandatoryFields);
            throw new CurrencyImportException(errorMsg);
        }
    }

    /**
     * Вытаскивает из полученного JSON список констант, а также ноды с описаниям валют.
     */
    private void deserializeCurrenciesJson() {
        ObjectMapper mapper = new ObjectMapper();
        try {
            JsonNode fieldsNode = mapper.readTree(currenciesJsonBody).get(FIELDS_ROOT_PROPERTY);
            currencyFields = mapper.convertValue(fieldsNode, TYPE_REF_FIELDS);
            currencyFields.put(CODE_FIELD, "код валюты");

            JsonNode currenciesNode = mapper.readTree(currenciesJsonBody).get(CURRENCIES_ROOT_PROPERTY);
            currencies = new ArrayList<>();
            if (currenciesNode.isArray()) {
                currenciesNode.iterator().forEachRemaining(currencies::add);
            }
        } catch (IOException ex) {
            throw new CurrencyImportException("can't deserialize currency json", ex);
        }
    }

    private static String getterName(String field) {
        if (field.equals(CODE_FIELD)) {
            return "getCode";
        } else {
            return "get" + upperCamel(field);
        }
    }

    /**
     * Большинство констант - денежные, для них возвращаем BigDecimal.
     * Здесь же захардкожен небольшой список исключений "имя" - "тип".
     */
    private Type getTypeByField(String field) {
        switch (field) {
            case CODE_FIELD:
                return CurrencyCode.class;
            case "LIST_ORDER":
            case "PRECISION_DIGIT_COUNT":
            case "YABS_RATIO":
            case "MIN_AUTOBUDGET_CLICKS_BUNDLE":
            case "AUTOBUDGET_CLICKS_BUNDLE_WARNING":
                return int.class;
            case "ISO_NUM_CODE":
                return Integer.class;
            case "BALANCE_CURRENCY_NAME":
                return String.class;
            case "MAX_AUTOBUDGET_CLICKS_BUNDLE":
                return long.class;
            default:
                return BigDecimal.class;
        }
    }

    private FieldSpec makeClassField(String field, JsonNode value) {
        Type fieldType = getTypeByField(field);

        FieldSpec.Builder builder = FieldSpec.builder(fieldType, field)
                .addModifiers(Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL);

        if (value.isNull()) {
            builder.initializer("null");
        } else if (fieldType.equals(BigDecimal.class)) {
            builder.initializer("new $T($S)", BigDecimal.class, value);
        } else if (fieldType.equals(String.class)) {
            builder.initializer("$S", value.textValue());
        } else if (fieldType.equals(int.class) || fieldType.equals(Integer.class)) {
            builder.initializer("$L", value);
        } else if (fieldType.equals(long.class) || fieldType.equals(Long.class)) {
            builder.initializer("$LL", value);
        } else if (fieldType.equals(CurrencyCode.class)) {
            builder.initializer("$T.$L", CurrencyCode.class, CurrencyCode.valueOf(value.textValue()));
        } else {
            throw new CurrencyImportException("unsupported field type " + fieldType.getTypeName());
        }

        return builder.build();
    }

    private MethodSpec.Builder makeGetterBuilder(String field) {
        return MethodSpec.methodBuilder(getterName(field))
                .addJavadoc(getDescription(field) + "\n")
                .addModifiers(Modifier.PUBLIC)
                .returns(getTypeByField(field));
    }

    private MethodSpec makeClassGetter(String field) {
        return makeGetterBuilder(field).addStatement("return $L", field).build();
    }

    private MethodSpec makeInterfaceGetter(String field) {
        return makeGetterBuilder(field)
                .addModifiers(Modifier.ABSTRACT)
                .addAnnotation(AnnotationSpec.builder(ConstantDescription.class)
                        .addMember("value", "$S", getDescription(field))
                        .build())
                .build();
    }

    private Stream<String> fields() {
        return currencyFields.keySet().stream().sorted();
    }

    private JavaFile makeInterface() {
        TypeSpec.Builder interfaceBuilder = TypeSpec.interfaceBuilder(classNamePrefix)
                .addAnnotation(generatedCurrenciesAnnotation)
                .addModifiers(Modifier.PUBLIC)
                .addJavadoc("Константы и параметры валюты\n\n")
                .addJavadoc("@see <a href=\"/direct/perl/protected/Currencies.pm\">Currencies</a>\n");

        fields().map(this::makeInterfaceGetter).forEach(interfaceBuilder::addMethod);

        return JavaFile
                .builder(interfacePackage, interfaceBuilder.build())
                .skipJavaLangImports(true)
                .indent(INDENT)
                .build();
    }

    private String getDescription(String field) {
        return currencyFields.getOrDefault(field, "значение константы " + field);
    }

    private static String upperCamel(String str) {
        return UPPER_UNDERSCORE.to(UPPER_CAMEL, str);
    }

    private JavaFile makeCurrencyClass(JsonNode currencyNode) {
        String code = currencyNode.get(CODE_FIELD).asText();
        String className = classNamePrefix + upperCamel(code);
        ClassName currencyClassName = ClassName.get(currenciesPackage, className);

        TypeSpec.Builder currencyBuilder = TypeSpec.classBuilder(currencyClassName)
                .addAnnotation(generatedCurrenciesAnnotation)
                .addModifiers(Modifier.PUBLIC)
                .addSuperinterface(ClassName.get(interfacePackage, classNamePrefix))
                .addJavadoc("Константы и параметры валюты " + code + "\n");

        FieldSpec instanceField = FieldSpec.builder(currencyClassName, "INSTANCE")
                .addModifiers(Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
                .initializer("new $T()", currencyClassName)
                .build();
        currencyBuilder.addField(instanceField);

        fields().map(f -> makeClassField(f, currencyNode.get(f))).forEach(currencyBuilder::addField);

        currencyBuilder.addMethod(MethodSpec.constructorBuilder().addModifiers(Modifier.PRIVATE).build());

        MethodSpec getInstanceMethod = MethodSpec.methodBuilder("getInstance")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .returns(currencyClassName)
                .addStatement("return INSTANCE")
                .build();
        currencyBuilder.addMethod(getInstanceMethod);

        fields().map(this::makeClassGetter).forEach(currencyBuilder::addMethod);

        MethodSpec toStringMethod = MethodSpec.methodBuilder("toString")
                .addModifiers(Modifier.PUBLIC)
                .returns(String.class)
                .addStatement("return \"Currency{\" + getCode() + \"}\"")
                .addAnnotation(Override.class)
                .build();
        currencyBuilder.addMethod(toStringMethod).build();

        return JavaFile
                .builder(currenciesPackage, currencyBuilder.build())
                .skipJavaLangImports(true)
                .indent(INDENT)
                .build();
    }

    /**
     * Вытаскивает из полученного JSON курсы валют и кладет в TreeMap, чтобы отсортировать по названию валют и датам.
     */
    private void deserializeRatesJson() {
        ObjectMapper mapper = new ObjectMapper();
        try {
            JsonNode ratesNode = mapper.readTree(ratesJsonBody).get("result");
            rates = new TreeMap<>();
            Iterator<Map.Entry<String, JsonNode>> iterCurr = ratesNode.fields();
            while (iterCurr.hasNext()) {
                Map.Entry<String, JsonNode> entryCurr = iterCurr.next();
                TreeMap<String, Pair<String, String>> ratesByDate = rates.computeIfAbsent(entryCurr.getKey(),
                        e -> new TreeMap<>());

                Iterator<Map.Entry<String, JsonNode>> iterDate = entryCurr.getValue().fields();
                while (iterDate.hasNext()) {
                    Map.Entry<String, JsonNode> entryDate = iterDate.next();
                    String date = entryDate.getKey();
                    String withNds = entryDate.getValue().get("with_nds").toString();
                    String withoutNds = entryDate.getValue().get("without_nds").toString();
                    ratesByDate.put(date, new ImmutablePair<>(withNds, withoutNds));
                }
            }

        } catch (IOException ex) {
            throw new CurrencyImportException("can't deserialize rates json", ex);
        }
    }

    private JavaFile makeRatesClass() {
        String className = "FixedRates";
        ClassName currencyClassName = ClassName.get(interfacePackage, className);

        TypeSpec.Builder classBuilder = TypeSpec.classBuilder(currencyClassName)
                .addAnnotation(generatedRatesAnnotation)
                .addModifiers(Modifier.PUBLIC)
                .addJavadoc("Курсы валют\n");

        TypeName typeList = ParameterizedTypeName.get(ClassName.get(List.class), ClassName.get(Rate.class));

        CodeBlock.Builder codeBuilder = CodeBlock.builder();
        codeBuilder.addStatement("private static final $T DATE_FORMAT = DateTimeFormatter.BASIC_ISO_DATE",
                DateTimeFormatter.class);

        FieldSpec.Builder dateFormat = FieldSpec.builder(ClassName.get(DateTimeFormatter.class), "DATE_FORMAT")
                .addModifiers(Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
                .initializer("DateTimeFormatter.BASIC_ISO_DATE");
        classBuilder.addField(dateFormat.build());

        FieldSpec.Builder beginOfTime = FieldSpec.builder(ClassName.get(LocalDate.class), "BEGIN_OF_TIME")
                .addModifiers(Modifier.STATIC, Modifier.FINAL)
                .initializer("LocalDate.ofEpochDay(0)");
        classBuilder.addField(beginOfTime.build());

        FieldSpec.Builder one = FieldSpec.builder(ClassName.get(Rate.class), "ONE")
                .addModifiers(Modifier.STATIC, Modifier.FINAL)
                .initializer("new Rate(BEGIN_OF_TIME, 1, 1)");
        classBuilder.addField(one.build());

        for (Map.Entry<String, TreeMap<String, Pair<String, String>>> entryCurr : this.rates.entrySet()) {
            String currInitializer = "ImmutableList.copyOf(new Rate[]{";

            for (Map.Entry<String, Pair<String, String>> entryDate : entryCurr.getValue().entrySet()) {
                String date = entryDate.getKey();
                String dateString = "0".equals(date) ? "BEGIN_OF_TIME, "
                        : "LocalDate.parse(\"" + date + "\", DATE_FORMAT), ";
                String withNds = entryDate.getValue().getLeft();
                String withoutNds = entryDate.getValue().getRight();

                currInitializer += "\n    new Rate(" + dateString + withNds + ", " + withoutNds + "),";
            }

            currInitializer += "\n})";

            String currencyCode = entryCurr.getKey();
            FieldSpec.Builder currency = FieldSpec.builder(typeList, currencyCode + "_HISTORY")
                    .addModifiers(Modifier.STATIC, Modifier.FINAL)
                    .initializer(currInitializer);
            classBuilder.addField(currency.build());
        }

        return JavaFile
                .builder(interfacePackage, classBuilder.build())
                .skipJavaLangImports(true)
                .indent(INDENT)
                .build();
    }

    private void writeJavaFiles() {
        try {
            makeInterface().writeTo(destRoot);
            for (JsonNode currency : currencies) {
                makeCurrencyClass(currency).writeTo(destRoot);
            }
            makeRatesClass().writeTo(destRoot);
        } catch (IOException | UnsupportedOperationException ex) {
            throw new CurrencyImportException("can't write java files", ex);
        }
    }

    /**
     * <ul>
     * <li>Загружает JSON с удаленного сервера ({@link #fetchRemoteJsons()})</li>
     * <li>Парсит JSON ({@link #deserializeCurrenciesJson()})</li>
     * <li>Валидирует загруженный JSON ({@link #validateData()})</li>
     * <li>Записывает сгенерированные Java-файлы в {@link #destRoot}
     * ({@link #writeJavaFiles()})</li>
     * <ul/>
     */
    public void doImport() {
        fetchRemoteJsons();
        deserializeCurrenciesJson();
        deserializeRatesJson();
        validateData();
        writeJavaFiles();
    }
}
