package ru.yandex.direct.dbschemagen;
/*
Оставлю здесь, для истории, причины, по которым не срослось с H2.

H2 в режиме совместимости с MySQL:
 - не умеет типы ENUM и SET
 - не умеет ON UPDATE, CHARACTER SET у полей, COMMENT у таблиц.
 - не умеет SET foreign_key_checks = 0
 - есть проблемы с именами индексов, например:
   - на KEY `i_cid` (`cid`) ругался: Index "I_CID" already exists
   - на UNIQUE KEY `cid` (`cid`,`metrika_counter`) ругался: Constraint "CID" already exists

Кроме этого, нельзя подсунуть джуку H2 под видом MySQL, потому что для анализа таблиц MySQL
jooq-codegen использует information_schema, которую H2 не эмулирует.

* Как идея на будущее: можно было бы создавать information_schema в H2 и подсовывать в
  jooq под видом MySQL, для этого нужно сконвертировать direct/trunk/db_schema в формат из
  которого можно получить как исходный SQL так и information_schema. Пробовать не стал - мало
  ли какие еще есть подводные камни у H2, а с докером - надежно.

Если же и заполнять базу H2 в режиме совместимости с MySQL, и запускать jooq-codegen
c использованием H2Database (а не MySQLDatabase), то сгенерированный код критически отличается
от того, что получается из мускуля, например, пропадают модификаторы unsigned, теряется регистр
символов в именах полей (в классах таблиц аттрибут CLIENTID ссылается на clientid вместо ClientID).
*/

import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.util.DefaultIndenter;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jooq.SQLDialect;
import org.jooq.codegen.GenerationTool;
import org.jooq.meta.ColumnDefinition;
import org.jooq.meta.Database;
import org.jooq.meta.Databases;
import org.jooq.meta.SchemaDefinition;
import org.jooq.meta.TableDefinition;
import org.jooq.meta.jaxb.Configuration;
import org.jooq.meta.jaxb.ForcedType;
import org.jooq.meta.jaxb.Generate;
import org.jooq.meta.jaxb.Generator;
import org.jooq.meta.jaxb.SchemaMappingType;
import org.jooq.meta.jaxb.Strategy;
import org.jooq.meta.jaxb.Target;

import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.mysql.MySQLDockerContainer;
import ru.yandex.direct.mysql.MySQLUtils;
import ru.yandex.direct.process.Docker;
import ru.yandex.direct.utils.io.FileUtils;

import static java.util.Arrays.asList;

public class DbSchemaGen {
    private final String packageNamespace;
    private final ArrayList<ForcedType> forcedTypes;
    private final JooqDbInfo db;

    public DbSchemaGen(String packageNamespace, JooqDbInfo db) {
        this.packageNamespace = packageNamespace;
        this.forcedTypes = new ArrayList<>();
        this.db = db;
    }

    public DbSchemaGen withForcedTypes(List<ForcedType> forcedTypes) {
        this.forcedTypes.addAll(forcedTypes);
        return this;
    }

    @SuppressWarnings("checkstyle:linelength")
    public void generate(String schemaName, Path generatedCodePath) {
        // jooq подклеивает к типу слово unsigned без пробела, см.
        // https://github.com/jOOQ/jOOQ/blob/version-3.7
        // .2/jOOQ-meta/src/main/java/org/jooq/util/mysql/MySQLTableDefinition.java#L96-L103
        // возможно, это баг.
        ForcedType[] forcedUnsignedTypes = {
                new ForcedType().withName("bigint").withTypes("int"),
                new ForcedType().withName("bigint").withTypes("intunsigned"),
                new ForcedType().withName("bigint").withTypes("mediumint"),
                new ForcedType().withName("bigint").withTypes("mediumintunsigned"),
                new ForcedType().withName("bigint").withTypes("smallint"),
                new ForcedType().withName("bigint").withTypes("smallintunsigned"),
                new ForcedType().withName("bigint").withTypes("tinyint"),
                new ForcedType().withName("bigint").withTypes("tinyintunsigned"),
                new ForcedType().withName("clob").withTypes("json")};

        Configuration configuration = new Configuration()
                .withJdbc(db.jooqJdbc().withSchema(schemaName))
                .withGenerator(new Generator()
                        .withStrategy(new Strategy()
                                .withName(DirectGeneratorStrategy.class.getCanonicalName()))
                        .withTarget(new Target()
                                .withPackageName(packageNamespace + "." + schemaName)
                                .withDirectory(generatedCodePath.toString()))
                        .withGenerate(new Generate()
                                .withJavaTimeTypes(true)
                        )
                        .withDatabase(db.jooqDatabase()
                                .withIncludes(".*")
                                .withExcludes("")
                                .withIntegerDisplayWidths(false)
                                .withInputSchema(schemaName)
                                .withForcedTypes(forcedTypes)
                                .withForcedTypes(forcedUnsignedTypes)
                        )
                );

        try {
            GenerationTool.generate(configuration);
        } catch (Exception ex) {
            throw new IllegalStateException(ex);
        }
    }

    public static List<String> guessArtificialBigintColumns(Connection conn, String schemaName) {
        List<String> artificialBigintColumns = new ArrayList<>();

        Database database = Databases.database(SQLDialect.MYSQL);
        database.setConnection(conn);
        database.setIncludes(new String[]{".*"});
        database.setSupportsUnsignedTypes(true);
        database.setConfiguredSchemata(Collections.singletonList(
                new SchemaMappingType()
                        .withInputSchema(schemaName)
                        .withOutputSchema(schemaName)
        ));

        for (SchemaDefinition schemaDefinition : database.getSchemata()) {
            for (TableDefinition tableDefinition : schemaDefinition.getTables()) {
                for (ColumnDefinition col : tableDefinition.getColumns()) {
                    if (col.getType().getType().equals("bigintunsigned") && !col.getName().contains("hash")) {
                        artificialBigintColumns.add(col.getQualifiedName());
                    }
                }
            }
        }

        return artificialBigintColumns;
    }

    public static void populate(
            Path schemaSqlDir, Connection connection, boolean splitDataByLines
    ) throws SQLException, IOException {
        MySQLUtils.loosenRestrictions(connection);
        populateFromFiles(schemaSqlDir, connection, "*.schema.sql", Collections::singletonList);
        populateFromFiles(schemaSqlDir, connection, "*.view.sql", Collections::singletonList);
        populateFromFiles(schemaSqlDir, connection, "*.procedure.sql", Collections::singletonList);

        Function<String, List<String>> dataSplitter = splitDataByLines
                ? x -> asList(x.split("\n"))
                : Collections::singletonList;
        populateFromFiles(schemaSqlDir, connection, "*.init-data.sql", dataSplitter);
        populateFromFiles(schemaSqlDir, connection, "*.data.sql", dataSplitter);
        populateFromFiles(schemaSqlDir, connection, "*.postprocess.sql", dataSplitter);
    }

    private static void populateFromFiles(
            Path schemaSqlDir, Connection connection, String glob,
            Function<String, List<String>> parser
    ) throws IOException {
        // сортируем файлы для консистентности
        List<Path> sortedFiles = new ArrayList<>();
        try (DirectoryStream<Path> sqlFilenames = Files.newDirectoryStream(schemaSqlDir, glob)) {
            sqlFilenames.forEach(sortedFiles::add);
        }
        Collections.sort(sortedFiles);

        for (Path filename : sortedFiles) {
            String content = FileUtils.slurp(filename);

            List<String> sqls = parser.apply(content);
            for (String sql : sqls) {
                try (Statement statement = connection.createStatement()) {
                    statement.executeUpdate(sql);
                } catch (SQLException exc) {
                    throw new IllegalStateException(
                            "Failed to execute query " + sql + " from file " + filename,
                            exc);
                }
            }
        }
    }

    public static void createSchema(Connection conn, String schemaName) throws SQLException {
        MySQLUtils.executeUpdate(
                conn,
                "CREATE SCHEMA " + MySQLUtils.quoteName(schemaName) + " DEFAULT CHARACTER SET utf8"
        );
    }

    /*  можно запускать прямо отсюда, добавив в "Program arguments:", в параметрах запуска (Edit configurations...) что нибудь вроде:
        -p ru.yandex.direct.dbschema -g /Users/hmepas/arc/arcadia/direct/libs/dbschema/src/main/java /Users/hmepas/arc/arcadia/direct/perl/db_schema/ppc
     */
    public static void main(String[] args) throws Exception {
        DbSchemaGenParams params = new DbSchemaGenParams();
        ParserWithHelp.parse(DbSchemaGen.class.getCanonicalName(), args, params);
        List<Path> schemaSqlDirs = params.schemaSqlDirs.stream().map(Paths::get).collect(Collectors.toList());

        runWithDocker(params, schemaSqlDirs, true);
    }

    public static void runWithDocker(DbSchemaGenParams params, List<Path> schemaSqlDirs, boolean splitDataByLines)
            throws SQLException, IOException, InterruptedException {
        try (MySQLDockerContainer db = new MySQLDockerContainer(new Docker())) {
            if (!params.guessArtificialBigintColumns) {
                DbSchemaGen generator = new DbSchemaGen(params.packageNamespace, new JooqDbInfo(db));
                for (Path schemaSqlDir : schemaSqlDirs) {
                    String schemaName = schemaSqlDir.getFileName().toString();
                    try (Connection conn = db.connect()) {
                        createSchema(conn, schemaName);
                        conn.setCatalog(schemaName);
                        populate(schemaSqlDir, conn, splitDataByLines);
                        generator.generate(schemaName, Paths.get(params.generatedCodePath));
                    }
                }

            } else {
                List<String> artificialBigintColumns = new ArrayList<>();
                for (Path schemaSqlDir : schemaSqlDirs) {
                    String schemaName = schemaSqlDir.getFileName().toString();
                    try (Connection conn = db.connect()) {
                        createSchema(conn, schemaName);
                        conn.setCatalog(schemaName);
                        populate(schemaSqlDir, conn, splitDataByLines);
                        artificialBigintColumns.addAll(guessArtificialBigintColumns(conn, schemaName));
                    }
                }
                new ObjectMapper()
                        .writer()
                        .with(new DefaultPrettyPrinter()
                                .withArrayIndenter(new DefaultIndenter("    ", DefaultIndenter.SYS_LF))
                        )
                        .writeValue(System.out, artificialBigintColumns);
            }
        }
    }
}
