package ru.yandex.direct.dbschemagen;

import java.nio.file.Path;
import java.sql.Connection;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.jooq.meta.jaxb.ForcedType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.mysql.MySQLDockerContainer;
import ru.yandex.direct.mysql.MySQLUtils;
import ru.yandex.direct.mysql.schema.DatabaseSchema;
import ru.yandex.direct.process.Docker;
import ru.yandex.direct.process.Processes;
import ru.yandex.direct.test.mysql.DirectMysqlDb;
import ru.yandex.direct.test.mysql.TestMysqlConfig;
import ru.yandex.direct.utils.GracefulShutdownHook;
import ru.yandex.direct.utils.io.TempDirectory;

public class DbSchemaUpdateSvn {
    private static final Logger logger = LoggerFactory.getLogger(DbSchemaUpdateSvn.class);

    private static final String[] DB_NAMES = {
            "ppc",
            "ppcdict",
            "monitor",
            "stubs",
            "chassis"
    };


    /**
     * Набор столбцов, генерируемых как {@link org.jooq.impl.SQLDataType#BIGINT}/{@link java.lang.Long}
     */
    private static final String[] ARTIFICIAL_BIGINT_COLUMNS = {
            "ppc.ab_segment_multiplier_values.ab_segment_multiplier_value_id",
            "ppc.ab_segment_multiplier_values.hierarchical_multiplier_id",
            "ppc.adgroups_dynamic.feed_id",
            "ppc.adgroups_mobile_content.mobile_content_id",
            "ppc.adgroups_performance.feed_id",
            "ppc.adgroups_text.feed_id",
            "ppc.agency_client_prove.agency_uid",
            "ppc.agency_client_prove.client_uid",
            "ppc.api_users_units_consumption.uid",
            "ppc.autobudget_cpa_alerts.cpa_deviation",
            "ppc.autobudget_cpa_alerts.apc_deviation",
            "ppc.autopay_settings.payer_uid",
            "ppc.balance_info_queue.operator_uid",
            "ppc.balance_info_queue.cid_or_uid",
            "ppc.banners_content_promotion_video.content_promotion_video_id",
            "ppc.banners_performance.creative_id",
            "ppc.banner_images_pool.imp_id",
            "ppc.banner_images_process_queue.imq_id",
            "ppc.banner_images_process_queue.operator_uid",
            "ppc.banner_images_process_queue_bid.imq_id",
            "ppc.banner_images_process_queue_bid.job_id",
            "ppc.banner_type_multiplier_values.banner_type_multiplier_value_id",
            "ppc.banner_type_multiplier_values.hierarchical_multiplier_id",
            "ppc.bids.id",
            "ppc.bids_arc.id",
            "ppc.bids_href_params.id",
            "ppc.bids_manual_prices.id",
            "ppc.bids_performance.perf_filter_id",
            "ppc.bids_phraseid_associate.bids_id",
            "ppc.bids_phraseid_history.id",
            "ppc.bm_reports.bs_report_id",
            "ppc.bs_resync_queue.id",
            "ppc.campaigns.uid",
            "ppc.campaigns.autobudget_goal_id",
            "ppc.camp_options.broad_match_goal_id",
            "ppc.clients_autoban.bids_count",
            "ppc.clients_options.balance_tid",
            "ppc.client_domains.record_id",
            "ppc.client_domains.sync_id",
            "ppc.client_domains_stripped.record_id",
            "ppc.content_promotion_video.content_promotion_video_id",
            "ppc.creative_banner_storage_sync.creative_id",
            "ppc.crm_queue.manager_uid",
            "ppc.currency_convert_queue.uid",
            "ppc.dbqueue_jobs.job_id",
            "ppc.dbqueue_jobs.uid",
            "ppc.dbqueue_job_archive.job_id",
            "ppc.dbqueue_job_archive.uid",
            "ppc.demography_multiplier_values.demography_multiplier_value_id",
            "ppc.demography_multiplier_values.hierarchical_multiplier_id",
            "ppc.eventlog.id",
            "ppc.eventlog.bids_id",
            "ppc.events.eid",
            "ppc.events.objectid",
            "ppc.events.objectuid",
            "ppc.events.uid",
            "ppc.feeds.feed_id",
            "ppc.filtered_feeds.feed_id",
            "ppc.geo_multiplier_values.geo_multiplier_value_id",
            "ppc.geo_multiplier_values.hierarchical_multiplier_id",
            "ppc.hierarchical_multipliers.hierarchical_multiplier_id",
            "ppc.infoblock_state.uid",
            "ppc.inventory_multiplier_values.hierarchical_multiplier_id",
            "ppc.inventory_multiplier_values.inventory_multiplier_value_id",
            "ppc.manager_got_servicing.uid",
            "ppc.manager_got_servicing.manager_uid",
            "ppc.mds_custom_names.id",
            "ppc.mds_custom_names.mds_id",
            "ppc.mds_metadata.id",
            "ppc.mobile_content.mobile_content_id",
            "ppc.mobile_content_icon_moderation_versions.mobile_content_id",
            "ppc.mobile_multiplier_values.mobile_multiplier_value_id",
            "ppc.mobile_multiplier_values.hierarchical_multiplier_id",
            "ppc.tablet_multiplier_values.tablet_multiplier_value_id",
            "ppc.tablet_multiplier_values.hierarchical_multiplier_id",
            "ppc.mod_object_version.obj_id",
            "ppc.mod_reasons.rid",
            "ppc.mod_reasons.id",
            "ppc.mod_resync_queue.id",
            "ppc.mod_resync_queue.object_id",
            "ppc.org_details.uid",
            "ppc.perf_creatives.creative_id",
            "ppc.perf_feed_categories.id",
            "ppc.perf_feed_categories.feed_id",
            "ppc.perf_feed_history.id",
            "ppc.perf_feed_history.feed_id",
            "ppc.perf_feed_vendors.id",
            "ppc.perf_feed_vendors.feed_id",
            "ppc.push_notifications_process.event_id",
            "ppc.retargeting_multiplier_values.retargeting_multiplier_value_id",
            "ppc.retargeting_multiplier_values.hierarchical_multiplier_id",
            "ppc.stat_reports.uid",
            "ppc.stat_reports.operator_uid",
            "ppc.trafaret_position_multiplier_values.trafaret_position_multiplier_value_id",
            "ppc.trafaret_position_multiplier_values.hierarchical_multiplier_id",
            "ppc.users_agency.uid",
            "ppc.users_api_options.uid",
            "ppc.users_captcha.uid",
            "ppc.users_surveys.uid",
            "ppc.users_surveys.ext_respondent_id",
            "ppc.user_campaigns_favorite.uid",
            "ppc.user_payment_transactions.id",
            "ppc.user_payment_transactions.uid",
            "ppc.wallet_campaigns.total_balance_tid",
            "ppc.wallet_payment_transactions.id",
            "ppc.wallet_payment_transactions.payer_uid",
            "ppc.wallet_payment_transactions.total_balance_tid",
            "ppc.weather_multiplier_values.weather_multiplier_value_id",
            "ppc.weather_multiplier_values.hierarchical_multiplier_id",
            "ppc.expression_multiplier_values.expression_multiplier_value_id",
            "ppc.expression_multiplier_values.hierarchical_multiplier_id",
            "ppc.warnings.id",
            "ppc.warnings_60_n.id",
            "ppc.warnings_nn.id",
            "ppc.warnplace.id",
            "ppcdict.api_app_certification_request.uid",
            "ppcdict.api_app_certification_request_history.uid",
            "ppcdict.api_domain_stat.sum_approx",
            "ppcdict.api_finance_tokens.uid",
            "ppcdict.api_queue_search_query.bs_id",
            "ppcdict.api_queue_stat.uid",
            "ppcdict.bad_karma_clients.uid",
            "ppcdict.bad_karma_clients.yandexuid",
            "ppcdict.bad_karma_clients.fuid",
            "ppcdict.inc_bsexport_iter_id.bsexport_iter_id",
            "ppcdict.inc_client_domains_record_id.client_domains_record_id",
            "ppcdict.inc_client_domains_stripped_record_id.client_domains_stripped_record_id",
            "ppcdict.inc_client_domains_sync_id.client_domains_sync_id",
            "ppcdict.inc_feed_id.feed_id",
            "ppcdict.inc_hierarchical_multiplier_id.hierarchical_multiplier_id",
            "ppcdict.inc_job_id.job_id",
            "ppcdict.inc_mobile_content_id.mobile_content_id",
            "ppcdict.inc_perf_filter_id.perf_filter_id",
            "ppcdict.inc_phid.phid",
            "ppcdict.lock_object.id",
            "ppcdict.lock_object.object_id",
            "ppcdict.mds_custom_names.id",
            "ppcdict.mds_custom_names.mds_id",
            "ppcdict.mds_metadata.id",
            "ppcdict.shard_creative_id.creative_id",
            "ppcdict.shard_login.uid",
            "ppcdict.shard_uid.uid",
            "ppcdict.ssl_certs.uid",
            "ppcdict.testusers.uid",
            "ppcdict.xls_reports.uid",
            "monitor.antispam_queue.request_id",
            "monitor.antispam_queue_id_log.request_id",
            "monitor.clients_geo_ip_cache.uid",
            "monitor.metrica_dead_domains.uid",
            "monitor.redirect_check_dict.id",
    };

    /**
     * Набор столбцов, генерируемых как {@link org.jooq.impl.SQLDataType#BIGINTUNSIGNED}/{@link org.jooq.types.ULong}.
     */
    private static final String[] ARTIFICIAL_BIGINTUNSIGNED_COLUMNS = {
            "ppc.perf_feed_categories.category_id",
            "ppc.perf_feed_categories.parent_category_id",
    };

    /**
     * Набор столбцов, генерируемых как {@link org.jooq.impl.SQLDataType#SMALLINTUNSIGNED}/{@link org.jooq.types.UShort}
     */
    private static final String[] ARTIFICIAL_SMALLINTUNSIGNED_COLUMNS = {
            "ppc.autobudget_alerts.problems",
    };

    private DbSchemaUpdateSvn() {
    }

    public static void main(String[] args) throws Exception {
        VcsWorker vcsWorker = VcsWorkerFactory.getWorker();

        DbSchemaUpdateSvnParams params = new DbSchemaUpdateSvnParams();
        ParserWithHelp.parse(DbSchemaUpdateSvn.class.getCanonicalName(), args, params);

        Path wcRoot = vcsWorker.getWorkingCopyRoot();
        if (!wcRoot.resolve(".arcadia.root").toFile().exists()) {
            Processes.exitWithFailure(
                    "Something is wrong with your arcadia checkout. Could not find file `.arcadia.root`."
            );
        }

        Path dbSchemaLibPath = wcRoot.resolve("direct/libs/dbschema");
        VcsReference libDbSchemaVcsReference = vcsWorker.createVcsReference(dbSchemaLibPath, params);
        if (!libDbSchemaVcsReference.isReproducible() && !params.forceGenerate) {
            Processes.exitWithFailure("" + dbSchemaLibPath + " must not have local modifications or mixed revisions.");
        }

        DirectMysqlDb testDb = new DirectMysqlDb(TestMysqlConfig.directConfig());

        try (
                GracefulShutdownHook ignored = new GracefulShutdownHook(Duration.ofMinutes(1));
                MySQLDockerContainer mysql = new MySQLDockerContainer(testDb.docker().createRunner(new Docker()));
                TempDirectory tmpDir = new TempDirectory("DbSchemaUpdateSvn")
        ) {
            String dbSchemaUrl = params.url;
            Path sourceInfoPath = dbSchemaLibPath.resolve("source_info.json");
            Optional<SourceInfo> prevSource = SourceInfo.load(sourceInfoPath);

            Path dbSchemaDir;
            GenericSourceInfo source;
            if (params.forceGenerate && !dbSchemaUrl.startsWith("svn+ssh://")) {
                // Разрешаем в качестве URL передавать путь до локальной папки со схемой
                dbSchemaDir = Path.of(params.url);
                source = new AbsentSourceInfo();
            } else {
                String dbSchemaRev = params.revision;
                dbSchemaDir = tmpDir.getPath().resolve("db_schema");

                /*
                Можно было бы сделать через svn.getUpdateClient().doCheckout(), но тогда возникают сложности
                с авторизацией, потому что SVNKit не умеет самостоятельно по схеме из урла определять правильный
                способ авторизации и всегда использует DefaultSVNAuthenticationManager.
                Посему, используем консольную утилитку.
                */
                logger.info("Checking out " + dbSchemaUrl + "@" + dbSchemaRev + "...");
                Processes.checkCall(
                        "svn",
                        "checkout",
                        "--revision", dbSchemaRev,
                        dbSchemaUrl,
                        dbSchemaDir.toString()
                );

                logger.info("Inspecting working copy: " + wcRoot + "...");
                source = new SourceInfo(
                        VcsWorkerFactory.getSvnWorker().createVcsReference(dbSchemaDir, params),
                        vcsWorker.createVcsReference(wcRoot.resolve("direct"), params),
                        mysql.getImage()
                );

                if (prevSource.isPresent() && prevSource.get().sameAs(source)) {
                    logger.info("Source code is not changed, re-generation is not necessary");
                    return;
                }
            }
            source.save(sourceInfoPath);

            /* ГЕНЕРАЦИЯ JOOQ-КЛАССОВ */

            List<ForcedType> forcedTypes = getForcedTypes();

            DbSchemaGen generator = new DbSchemaGen("ru.yandex.direct.dbschema", new JooqDbInfo(mysql))
                    .withForcedTypes(forcedTypes);

            try {
                for (Path schemaSqlDir : Arrays.stream(DB_NAMES).map(dbSchemaDir::resolve)
                        .collect(Collectors.toList())) {
                    String schemaName = schemaSqlDir.getFileName().toString();
                    try (Connection conn = mysql.connect(Duration.ofSeconds(20), Util::setMultiQueries)) {
                        DbSchemaGen.createSchema(conn, schemaName);
                        conn.setCatalog(schemaName);
                        DbSchemaGen.populate(schemaSqlDir, conn, true);
                        generator.generate(schemaName, dbSchemaLibPath.resolve("src/main/java"));
                    }
                }
            } catch (Exception exc) {
                logger.error("Generation failed, keeping SQL schema dir for further investigation: " + dbSchemaDir);
                tmpDir.keep();
                throw exc;
            }

            // Регистрируем изменения в vcs
            vcsWorker.registerChanges(dbSchemaLibPath);

            /* СОХРАНЕНИЕ ОБРАЗА БАЗЫ ДЛЯ ТЕСТОВ */

            // Имитируем шарды делая копии базы ppc
            try (Connection conn = mysql.connect()) {
                MySQLUtils.loosenRestrictions(conn);
                DatabaseSchema ppcSchema = DatabaseSchema.dump(conn, "ppc");
                logger.info("Cloning database ppc -> ppc1...");
                ppcSchema.rename("ppc1");
                ppcSchema.restore(conn);
                logger.info("Cloning database ppc -> ppc2...");
                ppcSchema.rename("ppc2");
                ppcSchema.restore(conn);
                logger.info("Dropping database ppc...");
                MySQLUtils.executeUpdate(conn, "DROP DATABASE ppc");
            }

            mysql.stop();

            var sourceTagParts = source.getTagString().split("/");
            String imageTag = "registry.yandex.net/direct/dbschema" + ":" + sourceTagParts[sourceTagParts.length - 1];

            logger.info("Snapshotting docker image " + imageTag + "...");
            testDb.docker().createImage(wcRoot, mysql, imageTag);

            if (source.isReproducible() || params.forceUpload) {
                logger.info("Uploading docker image " + imageTag + "...");
                mysql.getDocker().push(imageTag);

                if (!prevSource.isPresent()
                        || source.mysqlServerDiffers(prevSource.get())
                        || params.forceMysqlServerUpload
                ) {
                    logger.info("Uploading mysql server to sandbox...");
                    testDb.sandbox().uploadMysqlServer(wcRoot, mysql);
                }

                logger.info("Uploading mysql data to sandbox...");
                testDb.sandbox().uploadMysqlData(wcRoot, mysql);
            }

            if (!source.isReproducible()) {
                logger.warn("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
                logger.warn("");
                logger.warn("!!! WARNING !!!");
                logger.warn("");
                logger.warn("Changes made by this invocation of generator are not reproducible.");
                logger.warn("Please, do not commit these changes.");
                logger.warn("You can only use them locally for developing/debugging purposes.");
                logger.warn("");
                if (!params.forceUpload) {
                    logger.warn("Docker image created but not pushed to registry.");
                    logger.warn("Sandbox resources are not updated.");
                    logger.warn("");
                }
                logger.warn("For reproducible result you must run generator on clean arcadia checkout");
                logger.warn("(working copy must not have local modifications or mixed revisions).");
                logger.warn("");
                logger.warn("Please check " + sourceInfoPath);
                logger.warn("to find out whether your working copy has local modifications or mixed revisions.");
                logger.warn("");
                logger.warn("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
            }
        }
    }

    private static List<ForcedType> getForcedTypes() {
        Collection<ForcedType> forcedTypesLong =
                createForcedTypes(ARTIFICIAL_BIGINT_COLUMNS, "BIGINT");

        Collection<ForcedType> forcedTypesULong =
                createForcedTypes(ARTIFICIAL_BIGINTUNSIGNED_COLUMNS, "BIGINTUNSIGNED");

        Collection<ForcedType> forcedTypesUShort =
                createForcedTypes(ARTIFICIAL_SMALLINTUNSIGNED_COLUMNS, "SMALLINTUNSIGNED");

        List<ForcedType> forcedTypes = new ArrayList<>(forcedTypesLong);
        forcedTypes.addAll(forcedTypesULong);
        forcedTypes.addAll(forcedTypesUShort);
        Set<String> uniqExpressions =
                forcedTypes.stream().map(ForcedType::getExpression).collect(Collectors.toSet());
        if (uniqExpressions.size() != forcedTypesLong.size() + forcedTypesULong.size() + forcedTypesUShort.size()) {
            throw new IllegalStateException("Looks like there is overlapping in forced types. " +
                    "Please review your changes");
        }
        return forcedTypes;
    }

    /**
     * Генерирует набор {@link ForcedType}, переопределяющих sql-тип.
     *
     * @param columns           набор переопределяемых столбцов
     * @param forcedSqlTypeName необходимый sql-тип. Тип следует брать из {@link org.jooq.DataType}-констант
     *                          класса {@link org.jooq.impl.SQLDataType}
     * @return коллекцию настроенных ForcedTypes
     */
    private static Collection<ForcedType> createForcedTypes(String[] columns, String forcedSqlTypeName) {

        return Arrays.stream(columns)
                .map(column -> new ForcedType()
                        .withName(forcedSqlTypeName)
                        .withExpression(Pattern.quote(column))
                ).collect(Collectors.toList());
    }
}
