package ru.yandex.direct.dbschemagen;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.Path;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.exc.UnrecognizedPropertyException;

import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.HashingUtils;

public class SourceInfo implements GenericSourceInfo {
    private static final ObjectMapper MAPPER = new ObjectMapper();

    private VcsReference directDbSchemaSvnRef;
    private VcsReference dbSchemaGenRef;
    private String mysqlDockerImage;

    @JsonCreator
    public SourceInfo(
            @JsonProperty("directDbSchemaSvnRef") VcsReference directDbSchemaSvnRef,
            @JsonProperty("dbSchemaGenRef") VcsReference dbSchemaGenRef,
            @JsonProperty("mysqlDockerImage") String mysqlDockerImage
    ) {
        this.directDbSchemaSvnRef = directDbSchemaSvnRef;
        this.dbSchemaGenRef = dbSchemaGenRef;
        this.mysqlDockerImage = mysqlDockerImage;
    }

    @JsonValue
    public JsonNode toJson() {
        return MAPPER.createObjectNode()
                .putPOJO("directDbSchemaSvnRef", directDbSchemaSvnRef)
                .putPOJO("dbSchemaGenRef", dbSchemaGenRef)
                .put("mysqlDockerImage", mysqlDockerImage);
    }

    public VcsReference getDirectDbSchemaSvnRef() {
        return directDbSchemaSvnRef;
    }

    public VcsReference getDbSchemaGenRef() {
        return dbSchemaGenRef;
    }

    public String getMysqlDockerImage() {
        return mysqlDockerImage;
    }

    public static Optional<SourceInfo> load(Path path) throws IOException {
        try {
            return Optional.of(MAPPER.readValue(path.toFile(), SourceInfo.class));
        } catch (FileNotFoundException | UnrecognizedPropertyException ignored) {
            return Optional.empty();
        }
    }

    public void save(Path path) throws IOException {
        MAPPER.writerWithDefaultPrettyPrinter().writeValue(path.toFile(), this);
    }

    public boolean mysqlServerDiffers(GenericSourceInfo other) {
        if (!(other instanceof SourceInfo)) {
            return false;
        }
        // по хорошему нужно сравнивать айдишники а не имена образов
        return !mysqlDockerImage.equals(((SourceInfo) other).mysqlDockerImage);
    }

    public boolean differsFrom(SourceInfo other) {
        return mysqlServerDiffers(other)
                || !directDbSchemaSvnRef.sameAs(other.directDbSchemaSvnRef)
                || !dbSchemaGenRef.sameAs(other.dbSchemaGenRef);
    }

    public boolean sameAs(GenericSourceInfo other) {
        if (!(other instanceof SourceInfo)) {
            return false;
        }
        return !differsFrom((SourceInfo) other);
    }

    public String getTagString() {
        try {
            BigInteger md5 = new BigInteger(
                    HashingUtils.getMd5Hash(MAPPER.writerWithDefaultPrettyPrinter().writeValueAsBytes(this))
            );
            return "db" + directDbSchemaSvnRef.getRevision()
                    + (directDbSchemaSvnRef.hasLocalModifications() ? "dirty" : "")
                    + "-gen" + dbSchemaGenRef.getRevision()
                    + (dbSchemaGenRef.hasLocalModifications() ? "dirty" : "")
                    + "-" + mysqlDockerImage.replace(':', '-')
                    + "-" + md5.toString(16);
        } catch (JsonProcessingException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    public boolean isReproducible() {
        return directDbSchemaSvnRef.isReproducible() && dbSchemaGenRef.isReproducible();
    }
}
