package ru.yandex.direct.dbschemagen;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.Set;

import org.tmatesoft.svn.core.SVNDepth;
import org.tmatesoft.svn.core.wc.SVNClientManager;
import org.tmatesoft.svn.core.wc.SVNInfo;
import org.tmatesoft.svn.core.wc.SVNRevision;
import org.tmatesoft.svn.core.wc.SVNStatusType;

import ru.yandex.direct.process.Processes;

/**
 * todo javadoc
 */
public class SvnWorker implements VcsWorker {
    static final String SVN_ROOT = "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct";

    private static final SVNClientManager SVN = SVNClientManager.newInstance();

    @Override
    public VcsReference createVcsReference(Path sourceCodePath, DbSchemaUpdateVcsParams params) throws Exception {
        SVNInfo svnInfo = SVN.getWCClient().doInfo(sourceCodePath.toFile(), SVNRevision.WORKING);

        String url = svnInfo.getURL().toDecodedString();
        long revision = svnInfo.getRevision().getNumber();
        boolean mixedRevision;

        Set<Long> revisions = new HashSet<>();
        Holder<Boolean> hasLocalModificationsHolder = new Holder<>(false);

        SVN.getStatusClient().doStatus(
                sourceCodePath.toFile(),
                null,
                SVNDepth.INFINITY,
                false,
                true,
                false,
                false,
                status -> {
                    // Игнорируем неверсионированные файлы.
                    // В этом есть некоторая опасность. Неверсионированные файлы теоретически
                    // могут влиять на генератор. Но увы, отличить обычный аркадийный мусор от
                    // влияющих на генерацию изменений довольно сложно.
                    if (status.getRevision().getNumber() != -1) {
                        revisions.add(status.getRevision().getNumber());
                    }
                    if (
                            status.getContentsStatus() != SVNStatusType.STATUS_NONE
                                    && status.getContentsStatus() != SVNStatusType.STATUS_NORMAL
                    ) {
                        hasLocalModificationsHolder.value = true;
                    }
                },
                null
        );

        if (revisions.isEmpty() && !params.isForceGenerate()) {
            throw new IllegalStateException("Internal error: No versioned files found in " + sourceCodePath);
        }

        if (revisions.size() == 1 && !revisions.contains(revision) && !params.isForceGenerate()) {
            throw new IllegalStateException("Internal error: revisions mismatch in " + sourceCodePath
                    + ", [" + revision + "] != " + revisions
            );
        }

        mixedRevision = revisions.size() != 1;
        if (params.isForceGenerate()) {
            mixedRevision = false;
            hasLocalModificationsHolder.value = false;
        }
        return new VcsReference(url, revision, hasLocalModificationsHolder.value, mixedRevision);
    }

    @Override
    public Path getWorkingCopyRoot() throws Exception {
        SVNInfo svnInfo = SVN.getWCClient().doInfo(Paths.get(".").toFile(), SVNRevision.WORKING);
        return svnInfo.getWorkingCopyRoot().toPath();
    }

    @Override
    public void registerChanges(Path dbSchemaLibPath) throws Exception {
        SVN.getStatusClient().doStatus(
                dbSchemaLibPath.toFile(),
                null,
                SVNDepth.INFINITY,
                false,
                false,
                false,
                false,
                status -> {
                    if (status.getContentsStatus() == SVNStatusType.STATUS_MODIFIED) {
                        // svn и так знает про изменения и включит их в commit.
                        //noinspection UnnecessaryReturnStatement
                        return;
                    } else if (status.getContentsStatus() == SVNStatusType.STATUS_NONE) {
                        SVN.getWCClient().doAdd(
                                status.getFile(), false, false, false, SVNDepth.INFINITY, false, false
                        );
                    } else if (
                            status.getContentsStatus() == SVNStatusType.STATUS_MISSING
                                    // В какой-то момент svnkit стал возвращать NORMAL вместо MISSING
                                    // для удаленных файлов. Причина осталась загадкой, но, в целом, это не фатально,
                                    // до тех пор пока мы используем doStatus(reportAll=false).
                                    || status.getContentsStatus() == SVNStatusType.STATUS_NORMAL
                    ) {
                        // jooq codegen самостоятельно удаляет файлы при удалении таблиц,
                        // так что нам остается только зарегистрировать эти изменения в svn
                        SVN.getWCClient().doDelete(status.getFile(), false, false);
                    } else {
                        Processes.exitWithFailure(
                                "Unexpected SVN status: " + status.getContentsStatus().toString()
                                        + ", file: " + status.getFile().toString()
                        );
                    }
                },
                null
        );
    }

    private final class Holder<T> {
        T value;

        Holder(T value) {
            this.value = value;
        }
    }
}
