package ru.yandex.direct.ess.common.configuration;


import java.util.function.Supplier;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.scheduling.TaskScheduler;

import ru.yandex.direct.common.configuration.CommonConfiguration;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmIntegrationImpl;
import ru.yandex.direct.tvm.TvmIntegrationStub;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.kikimr.persqueue.auth.Credentials;

import static ru.yandex.direct.config.EssentialConfiguration.CONFIG_SCHEDULER_BEAN_NAME;

@Configuration
@Import({
        CommonConfiguration.class,

})
@ComponentScan(
        basePackages = {
                "ru.yandex.direct.ess.common"
        },
        excludeFilters = {
                @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION),
        }
)
public class EssCommonConfiguration {
    public static final String ESS_TVM_INTEGRATION_BEAN_NAME = "essTvmIntegration";
    public static final String ESS_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME = "essLogbrokerCredentialsSupplier";
    public static final String ESS_LOGBROKER_CLIENT_FACTORY_BEAN_NAME = "essLogbrokerClientFactory";

    @Bean(name = ESS_TVM_INTEGRATION_BEAN_NAME)
    public TvmIntegration essTvmIntegration(DirectConfig directConfig,
                                            @Qualifier(CONFIG_SCHEDULER_BEAN_NAME) TaskScheduler liveConfigChangeTaskScheduler) {
        boolean enabled = directConfig.findBoolean("ess.tvm.enabled").orElse(false);
        if (enabled) {
            return TvmIntegrationImpl.create(directConfig.getBranch("ess"), liveConfigChangeTaskScheduler);
        } else {
            return new TvmIntegrationStub();
        }
    }

    @Bean(name = ESS_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME)
    public Supplier<Credentials> getEssLogBrokerCredentialsSupplier(DirectConfig directConfig,
                                                                    @Qualifier(ESS_TVM_INTEGRATION_BEAN_NAME) TvmIntegration tvmIntegration) {
        TvmService tvmService = TvmService.fromStringStrict(directConfig.getString("ess.logbroker.tvm_service_name"));
        return () -> {
            String serviceTicket = tvmIntegration.getTicket(tvmService);
            return Credentials.tvm(serviceTicket);
        };
    }

    @Bean(name = ESS_LOGBROKER_CLIENT_FACTORY_BEAN_NAME)
    public LogbrokerClientFactoryFacade logbrokerClientCreatorFactory(
            @Qualifier(ESS_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME) Supplier<Credentials> logbrokerCredentialsSupplier) {
        return new LogbrokerClientFactoryFacade(logbrokerCredentialsSupplier);
    }
}
