package ru.yandex.direct.ess.common.logbroker;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderInitException;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.kikimr.persqueue.LogbrokerClientAsyncFactory;
import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.sync.SyncConsumerConfig;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.async.AsyncProducerConfig;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;

import static java.util.Collections.singletonList;

public class LogbrokerClientFactoryFacade {
    private static final Logger logger = LoggerFactory.getLogger(LogbrokerClientFactoryFacade.class);

    private final Supplier<Credentials> logbrokerCredentialsSupplier;

    private final Map<String, LogbrokerClientFactory> hostToLogbrokerClientRederFactory = new ConcurrentHashMap<>();
    private final Map<String, LogbrokerClientAsyncFactory> hostToLogbrokerClientWriterFactory =
            new ConcurrentHashMap<>();

    public LogbrokerClientFactoryFacade(Supplier<Credentials> logbrokerCredentialsSupplier) {
        this.logbrokerCredentialsSupplier = logbrokerCredentialsSupplier;
    }

    public Supplier<CompletableFuture<AsyncProducer>> createProducerSupplier(
            LogbrokerProducerProperties producerProperties,
            String sourceId) {
        AsyncProducerConfig producerConfig =
                getProducerConfig(producerProperties, sourceId);
        String host = producerProperties.getHost();
        LogbrokerClientAsyncFactory lcf = getLogbrokerClientWriterFactory(host);

        return () -> lcf.asyncProducer(producerConfig);
    }

    private AsyncProducerConfig getProducerConfig(LogbrokerProducerProperties logbrokerProducerProperties,
                                                  String sourceId) {
        byte[] sourceIdBytes = sourceId.getBytes(StandardCharsets.UTF_8);
        AsyncProducerConfig.Builder builder = AsyncProducerConfig
                .builder(logbrokerProducerProperties.getWriteTopic(), sourceIdBytes)
                .setCodec(logbrokerProducerProperties.getCompressionCodec())
                .setGroup(logbrokerProducerProperties.getGroup())
                .setCredentialsProvider(logbrokerCredentialsSupplier);

        return builder.build();
    }

    /**
     * Возвращает Supplier, создающий и инициализирующий SyncConsumer
     * Если не удалось создать или инициализировать - бросает LogbrokerReaderInitException, при этом если сессия
     * успевает открыться, то она закроется
     */

    public Supplier<SyncConsumer> createConsumerSupplier(LogbrokerConsumerProperties consumerProperties) {
        SyncConsumerConfig consumerConfig = getConsumerConfig(consumerProperties, logbrokerCredentialsSupplier);
        String host = consumerProperties.getHost();
        LogbrokerClientFactory lcf = getLogbrokerClientReaderFactory(host);
        long timeoutSec = consumerProperties.getReadDataTimeoutSec();
        return () -> {
            SyncConsumer syncConsumer = null;
            try {
                syncConsumer = lcf.syncConsumer(consumerConfig, timeoutSec, TimeUnit.SECONDS);
                var initResponse = syncConsumer.init();
                logger.info("Opened read session {}", initResponse.getSessionId());
            } catch (InterruptedException e) {
                closeConsumer(syncConsumer);
                Thread.currentThread().interrupt();
                throw new InterruptedRuntimeException(e);
            } catch (TimeoutException | RuntimeException e) {
                closeConsumer(syncConsumer);
                throw new LogbrokerReaderInitException(e);
            }
            return syncConsumer;
        };
    }

    private void closeConsumer(SyncConsumer syncConsumer) {
        if (Objects.nonNull(syncConsumer)) {
            try {
                syncConsumer.close();
            } catch (RuntimeException e) {
                logger.warn("Failed to close logbroker reader. Ignoring. Exception: {}", e);
            }
        }
    }

    private SyncConsumerConfig getConsumerConfig(LogbrokerConsumerProperties consumerProperties,
                                                 Supplier<Credentials> logBrokerCredentialsSupplier) {
        String readTopic = consumerProperties.getReadTopic();
        String consumerName = consumerProperties.getConsumerName();
        SyncConsumerConfig.Builder syncConsumerConfigBuilder =
                SyncConsumerConfig.builder(singletonList(readTopic), consumerName)
                        .setReadDataTimeout((int) consumerProperties.getReadDataTimeoutSec(), TimeUnit.SECONDS)
                        .setInitTimeout((int) consumerProperties.getInitTimeoutSec(), TimeUnit.SECONDS)
                        .setReadBufferSize(consumerProperties.getReadBufferSize())
                        .setCredentialsProvider(logBrokerCredentialsSupplier)
                        .configureReader(builder -> builder.setMaxCount(consumerProperties.getMaxCount()))
                        .configureSession(
                                configure -> configure.setGroups(new ArrayList<>(consumerProperties.getGroups()))
                        );
        if (consumerProperties.getReadDataOnlyAfterTimestampMs() != null) {
            syncConsumerConfigBuilder.configureReader(
                    configure -> configure.setReadDataOnlyAfterTimestampMs(consumerProperties.getReadDataOnlyAfterTimestampMs()));
        }
        if (consumerProperties.readOnlyNewData()) {
            syncConsumerConfigBuilder.configureReader(
                    configure -> configure.setReadDataOnlyAfterTimestampMs(System.currentTimeMillis()));
        }
        if (consumerProperties.getMaxSize() != null) {
            syncConsumerConfigBuilder.configureReader(
                    configure -> configure.setMaxSize(consumerProperties.getMaxSize()));
        }
        if (consumerProperties.getMaxUnconsumedReads() != null) {
            syncConsumerConfigBuilder.configureReader(
                    configure -> configure.setMaxUnconsumedReads(consumerProperties.getMaxUnconsumedReads()));
        }
        if (consumerProperties.getMaxTimeLagMs() != null) {
            syncConsumerConfigBuilder.configureReader(
                    configure -> configure.setMaxTimeLagMs(consumerProperties.getMaxTimeLagMs()));
        }
        return syncConsumerConfigBuilder.build();
    }

    private LogbrokerClientFactory getLogbrokerClientReaderFactory(String host) {
        if (hostToLogbrokerClientRederFactory.containsKey(host)) {
            return hostToLogbrokerClientRederFactory.get(host);
        } else {
            ProxyBalancer proxyBalancer = new ProxyBalancer(host);
            LogbrokerClientFactory lcf = new LogbrokerClientFactory(proxyBalancer);
            hostToLogbrokerClientRederFactory.put(host, lcf);
            return lcf;
        }
    }

    private LogbrokerClientAsyncFactory getLogbrokerClientWriterFactory(String host) {
        if (hostToLogbrokerClientWriterFactory.containsKey(host)) {
            return hostToLogbrokerClientWriterFactory.get(host);
        } else {
            ProxyBalancer proxyBalancer = new ProxyBalancer(host);
            LogbrokerClientAsyncFactory lcf = new LogbrokerClientAsyncFactory(proxyBalancer);
            hostToLogbrokerClientWriterFactory.put(host, lcf);
            return lcf;
        }
    }
}
