package ru.yandex.direct.ess.common.logbroker;

import java.util.Collections;
import java.util.List;


/**
 * Как устроено чтение.
 * <p>
 * Логброкерный клиент в фоне читает в свой {@code readBuffer} данные из партиций {@link #getGroups()}
 * топика {@link #getReadTopic()} как читатель {@link #getConsumerName()}.
 * Сервер нарезает сообщения на пачки ({code MessageBatch}, она потом коммитится целиком)
 * по {@link #getMaxCount()} штук суммарным размером {@link #getMaxSize()} (ограничение нестрогое,
 * может прийти на 1 сообщение больше).
 * <p>
 * При вызове (напрямую или нашей оберткой) метода {@link ru.yandex.kikimr.persqueue.consumer.SyncConsumer#read()}
 * — ответ берется из {@code readBuffer} или ожидается {@link #getReadDataTimeoutSec()} секунд до появления.
 * Если данные так и не появились, вернется {@code null}.
 * <p>
 * Умолчания значений смотри в {@link ru.yandex.kikimr.persqueue.consumer.internal.read.ReaderConfig}
 */
public interface LogbrokerConsumerProperties {
    /**
     * Хост логброкера, из которого будем читать данные
     */
    String getHost();

    /**
     * Время ожидания данных при чтении, если {@code readBuffer} пуст.
     */
    long getReadDataTimeoutSec();

    /**
     * Имя топика, который читаем
     */
    String getReadTopic();

    /**
     * Имя читателя, которым читаем (на него хранятся позиции чтения)
     */
    String getConsumerName();

    long getInitTimeoutSec();

    default int getRetries() {
        return 1;
    }

    default boolean readOnlyNewData() {
        return false;
    }

    default Long getReadDataOnlyAfterTimestampMs() {
        return null;
    }

    /**
     * Какие группы (связаны с партициями) хотим читать. Пустой список означает все.
     */
    default List<Integer> getGroups() {
        return Collections.emptyList();
    }

    /**
     * Ограничение на количество сообщений в одном {@code MessageBatch}
     */
    default int getMaxCount() {
        return 500;
    }

    /**
     * Ограничение на размер (в сжатом виде) сообщений в пачке.
     * Сервер может вернуть на одно сообщение больше.
     * <p>
     * Если не задано — поведение определено библиотекой LogBroker
     */
    default Integer getMaxSize() {
        return null;
    }

    /**
     * Не получать сообщения, у которых лаг больше указанного
     * <p>
     * Если не задано — поведение определено библиотекой LogBroker
     */
    default Integer getMaxTimeLagMs() {
        return null;
    }

    /**
     * Число сообщений, которое библиотека LogBroker может префетчить. В штуках!
     * <p>
     * Если в топике большие сообщение (> 50 Кбайт) — нужно делать маленькое число, иначе клиент может сожрать весь heap
     * прочитав много сообщений в свой {@code readBuffer}.
     * <p>
     * Размеры можно посмотреть в Solomon: {project=kikimr; service=pqproxy_writeInfo; sensor=MessageSizeOriginal}
     */
    default int getReadBufferSize() {
        return 1000;
    }

    /**
     * Ограничение на число сообщений, "ещё не полученных" из библиотеки LogBroker (методом {@code read}).
     * Не может быть больше {@link #getReadBufferSize()}.
     * <p>
     * Имеет смысл для топиков с тяжелыми сообщениями.
     * Если не задано - значение определено внутри библиотеки LogBroker
     */
    default Integer getMaxUnconsumedReads() {
        return null;
    }

}
