package ru.yandex.direct.ess.common.logbroker;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;

import ru.yandex.kikimr.persqueue.compression.CompressionCodec;

/**
 * Универсальная имплементация LogbrokerProducerProperties.
 * По задумке должна подойти в любом стандартном случае.
 * Default методы интерфейса можно не конфигурировать - тогда будет вызвана default имплементация.
 */
public class LogbrokerProducerPropertiesImpl implements LogbrokerProducerProperties {

    private final String writeTopic;
    private final String host;
    private final long readTimeoutSec;
    private final int retries;

    private final int group;

    private final boolean useCompressionCodec;
    private final CompressionCodec compressionCodec;

    private LogbrokerProducerPropertiesImpl(String writeTopic, String host, long readTimeoutSec, int retries,
                                            int group, boolean useCompressionCodec,
                                            @Nullable CompressionCodec compressionCodec) {
        this.host = host;
        this.writeTopic = writeTopic;
        this.readTimeoutSec = readTimeoutSec;
        this.retries = retries;
        this.group = group;

        this.useCompressionCodec = useCompressionCodec;
        this.compressionCodec = compressionCodec;
    }

    @Override
    public String getWriteTopic() {
        return writeTopic;
    }

    @Override
    public String getHost() {
        return host;
    }

    @Override
    public long getTimeoutSec() {
        return readTimeoutSec;
    }

    @Override
    public int getRetries() {
        return retries;
    }

    @Override
    public int getGroup() {
        return group;
    }

    @Override
    public CompressionCodec getCompressionCodec() {
        return useCompressionCodec ? compressionCodec : LogbrokerProducerProperties.super.getCompressionCodec();
    }

    @Override
    public String toString() {
        return "LogbrokerProducerPropertiesImpl{" +
                "host='" + host + '\'' +
                ", writeTopic='" + writeTopic + '\'' +
                ", readTimeoutSec=" + readTimeoutSec +
                ", retries=" + retries +
                ", group=" + group +
                ", useCompressionCodec=" + useCompressionCodec +
                ", compressionCodec=" + compressionCodec +
                '}';
    }

    public static LogbrokerProducerPropertiesImpl.Builder newBuilder() {
        return new LogbrokerProducerPropertiesImpl.Builder();
    }

    public static class Builder {
        private String writeTopic;
        private String host;
        private Long timeoutSec;
        private Integer retries;

        private Integer group;

        private boolean useCompressionCodec;
        private CompressionCodec compressionCodec;

        private Builder() {
        }

        public Builder setWriteTopic(String writeTopic) {
            this.writeTopic = writeTopic;
            return this;
        }

        public Builder setHost(String host) {
            this.host = host;
            return this;
        }

        public Builder setTimeoutSec(Long timeoutSec) {
            this.timeoutSec = timeoutSec;
            return this;
        }

        public Builder setRetries(Integer retries) {
            this.retries = retries;
            return this;
        }

        public Builder setGroup(Integer group) {
            this.group = group;
            return this;
        }

        public Builder setCompressionCodec(CompressionCodec compressionCodec) {
            this.useCompressionCodec = true;
            this.compressionCodec = compressionCodec;
            return this;
        }

        public LogbrokerProducerProperties build() {
            Preconditions.checkState(host != null, "forgotten host");
            Preconditions.checkState(writeTopic != null, "forgotten writeTopic");
            Preconditions.checkState(timeoutSec != null, "forgotten timeout");
            Preconditions.checkState(retries != null, "forgotten retries");
            Preconditions.checkState(group != null, "forgotten group");
            // compressionCodec разрешено быть null, будет вызвана дефолт имплементация
            return new LogbrokerProducerPropertiesImpl(writeTopic, host, timeoutSec, retries, group,
                    useCompressionCodec, compressionCodec);
        }
    }
}
