package ru.yandex.direct.ess.logicobjects.recomtracer;

import java.util.EnumMap;
import java.util.Map;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.jooq.Named;

import ru.yandex.direct.ess.common.models.BaseLogicObject;
import ru.yandex.direct.ess.common.utils.TablesEnum;

public class RecomTracerLogicObject extends BaseLogicObject {

    @JsonProperty("recommendation_type_id")
    private long recommendationTypeId;

    @JsonProperty("need_load")
    private boolean needLoad;

    @JsonProperty("table_to_load")
    private TablesEnum tableToLoad;

    @JsonProperty("primary_key")
    private Long primaryKey;

    @JsonProperty("recommendation_identifiers")
    private Map<RecommendationKeyIdentifier, Long> recommendationIdentifiers;

    @JsonProperty("additional_columns")
    private AdditionalColumns additionalColumns;

    @JsonCreator
    private RecomTracerLogicObject(
            @JsonProperty("recommendation_type_id") long recommendationTypeId,
            @JsonProperty("need_load") boolean needLoad,
            @JsonProperty("table_to_load") TablesEnum tableToLoad,
            @JsonProperty("primary_key") Long primaryKey,
            @JsonProperty("recommendation_identifiers") Map<RecommendationKeyIdentifier, Long> recommendationIdentifiers,
            @JsonProperty("additional_columns") AdditionalColumns additionalColumns

    ) {
        this.tableToLoad = tableToLoad;
        this.needLoad = needLoad;
        this.recommendationTypeId = recommendationTypeId;
        this.primaryKey = primaryKey;
        this.recommendationIdentifiers = recommendationIdentifiers;
        this.additionalColumns = additionalColumns;
    }

    public Long getRecommendationTypeId() {
        return recommendationTypeId;
    }

    public boolean isNeedLoad() {
        return needLoad;
    }

    public TablesEnum getTableToLoad() {
        return tableToLoad;
    }

    public Long getPrimaryKey() {
        return primaryKey;
    }

    public Long getRecommendationKeyIdentifier(RecommendationKeyIdentifier recommendationKeyIdentifier) {
        return recommendationIdentifiers.get(recommendationKeyIdentifier);
    }

    public boolean isRecommendationKeyIdentifierPresent(RecommendationKeyIdentifier recommendationKeyIdentifier) {
        return recommendationIdentifiers.containsKey(recommendationKeyIdentifier);
    }

    public AdditionalColumns getAdditionalColumns() {
        return additionalColumns;
    }

    @Override
    public String toString() {
        return "RecomTracerLogicObject{" +
                "recommendationTypeId=" + recommendationTypeId +
                ", needLoad=" + needLoad +
                ", tableToLoad=" + tableToLoad +
                ", primaryKey=" + primaryKey +
                ", recommendationIdentifiers=" + recommendationIdentifiers +
                ", additionalColumns=" + additionalColumns +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RecomTracerLogicObject object = (RecomTracerLogicObject) o;
        return recommendationTypeId == object.recommendationTypeId &&
                needLoad == object.needLoad &&
                Objects.equals(primaryKey, object.primaryKey) &&
                tableToLoad == object.tableToLoad &&
                Objects.equals(recommendationIdentifiers, object.recommendationIdentifiers) &&
                Objects.equals(additionalColumns, object.additionalColumns);
    }

    @Override
    public int hashCode() {
        return Objects.hash(recommendationTypeId, needLoad, tableToLoad, primaryKey, recommendationIdentifiers,
                additionalColumns);
    }

    public static class Builder {
        private TablesEnum tableToLoad;
        private Long recommendationTypeId;
        private Long primaryKey;
        private Map<RecommendationKeyIdentifier, Long> recommendationIdentifiers =
                new EnumMap<>(RecommendationKeyIdentifier.class);
        private AdditionalColumns additionalColumns = new AdditionalColumns();

        public Builder withTableToLoad(TablesEnum tableToLoad) {
            this.tableToLoad = tableToLoad;
            return this;
        }

        public Builder withRecommendationTypeId(Long recommendationTypeId) {
            this.recommendationTypeId = recommendationTypeId;
            return this;
        }

        public Builder withPrimaryKey(Long primaryKey) {
            this.primaryKey = primaryKey;
            return this;
        }

        public Builder addRecommendationKeyIdentifier(RecommendationKeyIdentifier recommendationKeyIdentifier,
                                                      Long value) {
            this.recommendationIdentifiers.put(recommendationKeyIdentifier, value);
            return this;
        }

        public Builder addAdditionalColumn(Named column, Object value) {
            this.additionalColumns.add(column, value);
            return this;
        }

        public RecomTracerLogicObject build() {
            boolean needLoad = Objects.nonNull(tableToLoad);
            return new RecomTracerLogicObject(recommendationTypeId, needLoad, tableToLoad, primaryKey,
                    recommendationIdentifiers, additionalColumns);
        }
    }
}
