package ru.yandex.direct.ess.fulltest;

import java.time.ZoneOffset;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.ess.client.EssClient;
import ru.yandex.direct.ess.common.models.BaseEssConfig;
import ru.yandex.direct.ess.common.models.BaseLogicObject;
import ru.yandex.direct.ess.common.models.EssGroup;
import ru.yandex.direct.ess.common.models.EssGroups;
import ru.yandex.direct.ess.fulltest.configuration.DebugEssConfiguration;
import ru.yandex.direct.ess.fulltest.configuration.comandline.DebugParams;
import ru.yandex.direct.ess.router.components.RulesProcessingService;
import ru.yandex.direct.ess.router.configuration.AppJcommanderConfiguration;
import ru.yandex.direct.ess.router.configuration.commandline.LogbrokerParams;
import ru.yandex.direct.ess.router.models.BinlogEventsFilter;
import ru.yandex.direct.ess.router.models.rule.AbstractRule;
import ru.yandex.direct.ess.router.models.rule.ProcessedObject;
import ru.yandex.direct.ess.router.models.rule.RuleProcessingResult;
import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.thread.ExecutionRuntimeException;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.ess.router.configuration.RouterApplicationConfiguration.ESS_SHARDS;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DebugEssRunner {

    private static final Logger logger = LoggingInitializer.getLogger(DebugEssRunner.class);

    private final LogbrokerBatchReader<BinlogEventWithOffset> logbrokerReader;
    private final EssClient essClient;

    @Autowired
    public DebugEssRunner(LogbrokerBatchReader<BinlogEventWithOffset> logbrokerReader,
                          EssClient essClient) {
        this.logbrokerReader = logbrokerReader;
        this.essClient = essClient;
    }

    public static void main(String[] args) {
        DebugParams debugParams = new DebugParams();
        LoggingInitializer
                .initialize(debugParams.routerParams.loggingInitializerParams, DebugEssRunner.class.getCanonicalName());
        AppJcommanderConfiguration.initializeParameters(debugParams.routerParams);
        ParserWithHelp.parse(DebugEssRunner.class.getCanonicalName(), args, debugParams);

        // Для запуска дебага нет возможности делать коммит, можно указывать параметр read-new-data - тогда будут
        // обрабатываться события, начиная с момента запуска приложения
        debugParams.routerParams.logbrokerParams.logbrokerNoCommit = true;

        debugParams.validate();

        try (AnnotationConfigApplicationContext ctx = new AnnotationConfigApplicationContext(
                DebugEssConfiguration.class)) {
            ctx.getBean(DebugEssRunner.class).runEss(ctx, debugParams);
        }
    }

    private void runEss(AnnotationConfigApplicationContext ctx, DebugParams debugParams) {
        LogbrokerParams logbrokerParams = debugParams.routerParams.logbrokerParams;

        Set<AbstractRule<BaseLogicObject>> selectedRules = new HashSet<>();
        if (!debugParams.essConfigClassNames.isEmpty()) {
            selectedRules.addAll(getRulesByEssConfigNames(ctx, debugParams.essConfigClassNames));
        } else {
            selectedRules.addAll(getRulesByEssGroup(ctx, debugParams.essGroup));
        }
        checkState(!selectedRules.isEmpty(), "No rules were found");

        List<Class<? extends AbstractRule>> ruleClasses = mapList(selectedRules, AbstractRule::getClass);
        logger.info("Found rules: {}", mapList(ruleClasses, Class::getCanonicalName));

        if (StringUtils.isNotBlank(debugParams.routerParams.essTag)) {
            logger.info("Using ess tag \"{}\"", debugParams.routerParams.essTag);
        }

        RulesProcessingService rulesProcessingService =
                createRulesProcessingService(ctx, logbrokerParams, selectedRules);

        List<Integer> shards = ctx.getBean(ESS_SHARDS, List.class);

        List<EssChain> essChains = createEssChains(ctx, ruleClasses, shards);
        Processor processor = new Processor(rulesProcessingService, essChains);

        int workingTimeSec = debugParams.workingTimeSec * 1000;
        long startTime = System.currentTimeMillis();
        try {
            while (!Thread.currentThread().isInterrupted() && System.currentTimeMillis() - startTime <= workingTimeSec) {
                try {
                    logbrokerReader.fetchEvents(processor);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        } finally {
            essChains.forEach(EssChain::finishProcessors);
        }
    }

    private List<AbstractRule<BaseLogicObject>> getRulesByEssConfigNames(ApplicationContext ctx,
                                                                         List<String> essConfigClassNames) {
        List<Class<? extends BaseEssConfig>> essConfigs = essClient.getEssConfigsClasses().stream()
                .filter(essConfig ->
                        essConfigClassNames.stream().anyMatch(r -> essConfig.getCanonicalName().endsWith(r)))
                .collect(toList());

        return getRulesByConfigs(ctx, essConfigs);
    }

    private List<AbstractRule<BaseLogicObject>> getRulesByEssGroup(ApplicationContext ctx, String essGroupName) {
        var targetGroup = EssGroup.valueOf(essGroupName.toUpperCase());

        List<Class<? extends BaseEssConfig>> configs = filterList(essClient.getEssConfigsClasses(), configClass -> {
            var essGroupsAnnotation = configClass.getAnnotation(EssGroups.class);
            return essGroupsAnnotation != null &&
                    StreamEx.of(essGroupsAnnotation.value()).anyMatch(group -> group == targetGroup);
        });

        return getRulesByConfigs(ctx, configs);
    }

    private List<AbstractRule<BaseLogicObject>> getRulesByConfigs(ApplicationContext ctx,
                                                                  List<Class<? extends BaseEssConfig>> configs) {
        return mapList(configs, conf -> EssChainUtils.getRule(ctx, conf));
    }

    private RulesProcessingService createRulesProcessingService(ApplicationContext ctx, LogbrokerParams logbrokerParams,
                                                                Set<? extends AbstractRule> selectedRules) {
        BinlogEventsFilter binlogEventsFilter = ctx.getBean(BinlogEventsFilter.class);
        if (logbrokerParams.getReadOnlyDataCreatedAfterTimestampMs() != null) {
            long startTime = logbrokerParams.getReadOnlyDataCreatedAfterTimestampMs() / 1000;
            binlogEventsFilter.addFilter(event ->
                    event.getEvent().getUtcTimestamp().toEpochSecond(ZoneOffset.UTC) >= startTime);
        }
        var defaultCondition = ctx.getBean(TypicalEnvironment.class);
        var ruleToCondition = StreamEx.of(selectedRules)
                .mapToEntry(r -> r.getClass(), r -> defaultCondition)
                .toMap();
        return new RulesProcessingService(List.copyOf(selectedRules), binlogEventsFilter,
                ctx.getBean(PpcPropertiesSupport.class), ruleToCondition);
    }

    private List<EssChain> createEssChains(ApplicationContext ctx, List<Class<? extends AbstractRule>> ruleClasses,
                                           List<Integer> shards) {
        return mapList(ruleClasses, ruleClass -> {
            var logicProcessorSupplier = EssChainUtils.getBaseLogicProcessorSupplierForRule(ctx, ruleClass);
            var essChain = new EssChain(ruleClass, logicProcessorSupplier);
            essChain.initProcessors(shards);
            return essChain;
        });
    }

    private class Processor implements Interrupts.InterruptibleConsumer<List<BinlogEventWithOffset>> {
        private final RulesProcessingService rulesProcessingService;
        private final Map<String, EssChain> essChains;

        public Processor(RulesProcessingService rulesProcessingService, List<EssChain> essChains) {
            this.rulesProcessingService = rulesProcessingService;
            this.essChains = listToMap(essChains, EssChain::getTopic);
        }

        @Override
        public void accept(List<BinlogEventWithOffset> value) {
            CompletableFuture[] futures = rulesProcessingService
                    .processEvents(value, CompletableFuture.completedFuture(null))
                    .map(cf -> cf.thenAccept(this::handleRuleProcessingResult))
                    .toArray(CompletableFuture[]::new);

            try {
                CompletableFuture.allOf(futures).get();
            } catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                throw new InterruptedRuntimeException(ex);
            } catch (ExecutionException ex) {
                throw new ExecutionRuntimeException(ex);
            }
        }

        private void handleRuleProcessingResult(RuleProcessingResult processingResult) {
            var topic = processingResult.getTopic();
            var essChain = essChains.get(topic);
            var logicObjectsConverter = essChain.getLogicObjectsConverter();

            Map<Integer, List<ProcessedObject>> partitionsToProcessedObjects =
                    processingResult.getGroupedLogicObjectsByPartitions();

            partitionsToProcessedObjects.entrySet().parallelStream()
                    .forEach(partitionToProcessedObjects -> {
                        //шард - это партиция + 1
                        int shard = partitionToProcessedObjects.getKey() + 1;
                        List<ProcessedObject> objects = partitionToProcessedObjects.getValue();

                        List<BaseLogicObject> logicObjects = objects.stream()
                                .map(ProcessedObject::getLogicObjectWithSystemInfo)
                                .map(logicObjectsConverter::fromJson)
                                .flatMap(logicObjectWithSystemInfo ->
                                        logicObjectWithSystemInfo.getLogicObjectsList().stream())
                                .collect(toList());

                        var processor = essChain.getProcessorForShard(shard);
                        logger.info("sending {} logic objects from shard {} to processor {}",
                                logicObjects.size(), shard, processor.getLogicProcessName());
                        processor.processBaseObjects(logicObjects);
                    });
        }
    }
}
