package ru.yandex.direct.ess.fulltest;

import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.ess.common.converter.LogicObjectWithSystemInfoConverter;
import ru.yandex.direct.ess.common.models.BaseEssConfig;
import ru.yandex.direct.ess.common.models.BaseLogicObject;
import ru.yandex.direct.ess.router.models.rule.AbstractRule;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessorNotScheduled;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static java.util.stream.Collectors.toMap;

@ParametersAreNonnullByDefault
public class EssChain {
    private final Class<? extends AbstractRule> ruleClass;
    private final Supplier<? extends BaseLogicProcessorNotScheduled> logicProcessorSupplier;
    private final BaseEssConfig essConfig;

    private Map<Integer, BaseLogicProcessorNotScheduled<?>> shardsToProcessors;

    public EssChain(Class<? extends AbstractRule> ruleClass,
                    Supplier<? extends BaseLogicProcessorNotScheduled> logicProcessorSupplier) {
        this.ruleClass = ruleClass;
        this.logicProcessorSupplier = logicProcessorSupplier;
        essConfig = getEssConfig();
    }

    private BaseEssConfig getEssConfig() {
        BaseEssConfig result;
        try {
            result = ruleClass.getAnnotation(EssRule.class).value()
                    .getDeclaredConstructor().newInstance();
        } catch (ReflectiveOperationException e) {
            throw new IllegalStateException("Can't create new instance of ess config class", e);
        }
        return result;
    }

    public Map<Integer, BaseLogicProcessorNotScheduled<?>> initProcessors(List<Integer> shards) {
        shardsToProcessors = shards.stream()
                .map(shard -> {
                    BaseLogicProcessorNotScheduled baseLogicProcessor = logicProcessorSupplier.get();
                    baseLogicProcessor.initialize(TaskParametersMap.of("shard", String.valueOf(shard)));
                    return baseLogicProcessor;
                })
                .collect(toMap(
                        DirectShardedJob::getShard,
                        baseLogicProcessor -> baseLogicProcessor
                ));

        return shardsToProcessors;
    }

    public LogicObjectWithSystemInfoConverter<? extends BaseLogicObject> getLogicObjectsConverter() {
        return new LogicObjectWithSystemInfoConverter<>(essConfig.getLogicObject());
    }

    public String getTopic() {
        return essConfig.getTopic();
    }

    public void finishProcessors() {
        shardsToProcessors.values().forEach(BaseLogicProcessorNotScheduled::finish);
    }

    public BaseLogicProcessorNotScheduled getProcessorForShard(int shard) {
        return shardsToProcessors.get(shard);
    }
}
