package ru.yandex.direct.ess.fulltest.configuration;

import java.util.List;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;

import ru.yandex.direct.binlogbroker.logbroker_utils.lock.ConsumerLockHolder;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.ess.client.EssClientConfiguration;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerProperties;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerPropertiesImpl;
import ru.yandex.direct.ess.router.configuration.AppJcommanderConfiguration;
import ru.yandex.direct.ess.router.configuration.RouterApplicationConfiguration;
import ru.yandex.direct.ess.router.configuration.commandline.AppParams;
import ru.yandex.direct.ess.router.configuration.commandline.LogbrokerParams;
import ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestFilter;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationStageEnum;
import ru.yandex.direct.logicprocessor.processors.moderation.VoidModerationRequestFilter;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.ytwrapper.client.YtClusterTypesafeConfigProvider;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.compression.CompressionCodec;

import static ru.yandex.direct.ess.router.configuration.RouterApplicationConfiguration.CONSUMER_NAME_BEAN;
import static ru.yandex.direct.ess.router.configuration.RouterApplicationConfiguration.ESS_SHARDS;
import static ru.yandex.direct.ess.router.configuration.RouterApplicationConfiguration.ROUTER_CHUNK_ID;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_ADGROUP_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_AD_IMAGE_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNERSTORAGE_CREATIVE_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_VCARD_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_VIDEO_ADDITION_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CALLOUT_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CAMPAIGN_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CANVAS_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CPM_GEO_PIN_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CPM_VIDEO_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_DEFAULT_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_DISPLAYHREFS_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_HTML5_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_IMAGE_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_INTERNAL_BANNER_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_MOBILE_CONTENT_ICON_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_PROMO_EXTENSION_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SITELINKS_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_CAMPAIGN_COPY_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_DELETION_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_TEXT_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_TURBOLANDING_REQUEST;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_TVM_INTEGRATION_BEAN_NAME;
import static ru.yandex.direct.logicprocessor.configuration.EssLogicProcessorConfiguration.MODERATION_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Configuration
@Import({
        EssLogicProcessorConfiguration.class,
        AppJcommanderConfiguration.class,
        RouterApplicationConfiguration.class,
        EssClientConfiguration.class,
})
@ComponentScan(
        basePackages = {
                "ru.yandex.direct.ess.fulltest",
                "ru.yandex.direct.logicprocessor"
        },
        excludeFilters = {
                @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
        }
)
public class DebugEssConfiguration {

    @Primary
    @Bean(name = ESS_SHARDS)
    public List<Integer> essShards(
            LogbrokerParams logbrokerParams,
            ShardHelper shardHelper
    ) {
        var totalShards = shardHelper.dbShards();
        // если в параметрах явно задан список шардов - берём его
        if (logbrokerParams.partitions != null && !logbrokerParams.partitions.isEmpty()) {
            return logbrokerParams.partitions.stream()
                    .filter(totalShards::contains)
                    .collect(Collectors.toList());
        } else {
            return totalShards;
        }
    }

    @Primary
    @Bean(name = ROUTER_CHUNK_ID)
    public String routerChunkId() {
        return "debugChunk";
    }

    @Bean
    public ConsumerLockHolder consumerLockHolder(
            DirectConfig directConfig,
            @Value("${debug-ess.yt_lock.cluster_name}") String clusterName,
            @Value("${debug-ess.yt_lock.lock_path_prefix}") String lockPathPrefixName,
            ConfigurableApplicationContext configurableApplicationContext) {
        var ytClusterConfigProvider = new YtClusterTypesafeConfigProvider(directConfig.getConfig().getConfig("yt"));
        var possibleConsumers = directConfig.getStringList("debug-ess.logbroker.consumers");
        return new ConsumerLockHolder(ytClusterConfigProvider, clusterName, lockPathPrefixName, possibleConsumers,
                connectionLostHandler(configurableApplicationContext));
    }

    @Primary
    @Bean(CONSUMER_NAME_BEAN)
    public String consumerName(ConsumerLockHolder consumerLockHolder) {
        var consumerNameOptional = consumerLockHolder.getConsumerName();
        if (consumerNameOptional.isEmpty()) {
            throw new IllegalStateException("All consumers are busy");
        }
        return consumerNameOptional.get();
    }

    @Bean(name = MODERATION_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME)
    @Primary
    public Supplier<Credentials> getModerationLogbrokerCredentialsSupplier(DirectConfig directConfig,
                                                                           AppParams appParams,
                                                                           @Qualifier(MODERATION_TVM_INTEGRATION_BEAN_NAME) TvmIntegration tvmIntegration) {
        var moderationStage = ModerationStageEnum.fromValue(appParams.essTag);
        var tvmServiceName = moderationStage == null ?
                directConfig.getString("moderation_service.logbroker.tvm_service_name")
                : directConfig.getString(String.format("moderation_service.debug.logbroker.%s.tvm_service_name",
                moderationStage.getValue()));
        var tvmService = TvmService.fromStringStrict(tvmServiceName);
        return () -> {
            String serviceTicket = tvmIntegration.getTicket(tvmService);
            return Credentials.tvm(serviceTicket);
        };
    }

    @Primary
    @Bean
    public ModerationRequestFilter moderationRequestFilter() {
        return new VoidModerationRequestFilter();
    }

    private Consumer<Exception> connectionLostHandler(ConfigurableApplicationContext configurableApplicationContext) {
        return ex -> {
            configurableApplicationContext.close();
            System.exit(1);
        };
    }

    private List<LogbrokerProducerProperties> buildProducersPropertiesFromConfig(DirectConfig logbrokerConf,
                                                                                 DirectConfig stageConfig) {
        CompressionCodec compressionCodec =
                CompressionCodec.valueOf(stageConfig.findString("compression_codec")
                        .orElse("gzip").toUpperCase());

        return mapList(logbrokerConf.getIntList("group"), group ->
                LogbrokerProducerPropertiesImpl.newBuilder()
                        .setHost(stageConfig.getString("host"))
                        .setWriteTopic(stageConfig.getString("topic"))
                        .setTimeoutSec(logbrokerConf.getLong("data_timeout"))
                        .setRetries(logbrokerConf.getInt("retries"))
                        .setGroup(group)
                        .setCompressionCodec(compressionCodec)
                        .build());
    }

    private List<LogbrokerProducerProperties> makeProperties(AppParams params, DirectConfig conf, String blockName) {
        ModerationStageEnum moderationStage = ModerationStageEnum.fromValue(params.essTag);

        DirectConfig logbrokerConf = conf.getBranch("moderation_service.logbroker");

        DirectConfig stageConfig = moderationStage == null ?
                logbrokerConf.getBranch(blockName)
                : conf.getBranch(String.format("moderation_service.debug.logbroker.%s.%s",
                moderationStage.getValue(), blockName));

        return buildProducersPropertiesFromConfig(logbrokerConf, stageConfig);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_ADGROUP_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesAdGroupRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "adgroup_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_DEFAULT_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesDefaultRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "banner_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_TURBOLANDING_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesTurbolandingsRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "turbolanding_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_DISPLAYHREFS_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesDisplayhrefsRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "displayhrefs_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SITELINKS_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesSitelinksRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "sitelinks_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_IMAGE_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesImageRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "images_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CPM_VIDEO_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCpmVideoRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "cpm_video_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_AD_IMAGE_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesAdImageRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "ad_image_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_TEXT_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesTextRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "text_banner_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_DELETION_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesSpecialDeletionRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "special_deletion_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_CAMPAIGN_COPY_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesSpecialCampaignCopyRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "special_campaign_copy_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CALLOUT_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCalloutRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "callout_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_VCARD_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesVcardRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "banner_vcard_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_VIDEO_ADDITION_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesVideoAdditionRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "banner_video_addition_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CPM_GEO_PIN_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCpmGeoPinRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "cpm_geo_pin_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_PROMO_EXTENSION_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesPromoExtensionRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "promo_extension_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNERSTORAGE_CREATIVE_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesBannerstorageCreativeRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "bannerstorage_creative_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_HTML5_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesHtml5Request(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "html5_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CANVAS_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCanvasRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "canvas_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_INTERNAL_BANNER_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesInternalBannerRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "internal_banner_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_MOBILE_CONTENT_ICON_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesMobileContentIconRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "mobile_content_icon_moderation_request");
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CAMPAIGN_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCampaignRequest(
            AppParams params, DirectConfig conf) {
        return makeProperties(params, conf, "campaign_moderation_request");
    }
}
