package ru.yandex.direct.logicprocessor.common;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderCloseException;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader;
import ru.yandex.direct.ess.common.models.BaseEssConfig;
import ru.yandex.direct.ess.common.models.BaseLogicObject;
import ru.yandex.direct.ess.common.models.LogicObjectListWithInfo;
import ru.yandex.direct.ess.common.utils.EssCommonUtils;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.logicprocessor.components.LogicObjectLogbrokerReader;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Base class for custom logic processors.
 * Not scheduled. You need to add @Hourglass and @HourglassDaemon annotations in inheritor classes.
 * Each inheritors should define the annotation {@link EssLogicProcessor}
 * , call super constructor with autowired EssLogicProcessorContext
 * and define a void method process(List<BaseLogicObject> logicObjects)
 */
public abstract class BaseLogicProcessorNotScheduled<T extends BaseLogicObject> extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(BaseLogicProcessor.class);

    private final EssLogicProcessorContext essLogicProcessorContext;
    private final BaseEssConfig baseEssConfig;
    private final Processor processor;
    private final long criticalEssProcessTimeSec;
    private final String logicProcessName;
    private final EssLogicObjectsBlacklist blacklist;
    private final AtomicLong maxTimestamp;

    private LogicProcessorMonitoring logicProcessorMonitoring;
    private LogbrokerBatchReader<LogicObjectListWithInfo<T>> logbrokerReader;
    private boolean isInitialized = false;
    private Labels solomonRegistryLabels;
    private MetricRegistry metricRegistry;

    protected BaseLogicProcessorNotScheduled(EssLogicProcessorContext essLogicProcessorContext) {
        EssLogicProcessor essLogicProcessor =
                this.getClass().getAnnotation(EssLogicProcessor.class);
        this.essLogicProcessorContext = essLogicProcessorContext;
        this.processor = new Processor();

        Class<? extends BaseEssConfig> configClazz = essLogicProcessor.value();
        try {
            baseEssConfig = configClazz.getDeclaredConstructor().newInstance();
        } catch (Exception e) {
            throw new LogicProcessorException(e);
        }

        this.criticalEssProcessTimeSec = baseEssConfig.getCriticalEssProcessTime().getSeconds();
        this.logicProcessName = baseEssConfig.getLogicProcessName();
        this.blacklist = essLogicProcessorContext.getBlacklist();
        this.maxTimestamp = new AtomicLong(0);
    }

    public String getLogicProcessName() {
        return logicProcessName;
    }

    /**
     * Method that processing of logicObjects. When the messages read from logbroker are more than
     * {@link BaseEssConfig#getRowsThreshold()}
     * or the time of reading messages are more than {@link BaseEssConfig#getTimeToReadThreshold()}
     * all read messages converts to {@link BaseEssConfig#getLogicObject()} class and this method starts
     *
     * @param logicObjects list of BaseLogicObjects read from logbroker topic {@link BaseEssConfig#getTopic()}
     */
    public abstract void process(List<T> logicObjects);

    protected void initialize() {
    }

    protected MetricRegistry getMetricRegistry() {
        checkArgument(metricRegistry != null, "initialize not called");
        return metricRegistry;
    }

    private void initializeInternal() {
        initializeInternal(true);
    }

    /**
     * @param needLogbroker - используется при запуске всей ess цепочки, в этом случае чтения из логброкера не
     *                      происходит и создавать reader не нужно
     */
    private void initializeInternal(boolean needLogbroker) {
        if (!isInitialized) {
            solomonRegistryLabels =
                    Labels.of("ess_processor", logicProcessName, "shard", String.valueOf(getShard()));
            metricRegistry = SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels);
            this.logicProcessorMonitoring =
                    new LogicProcessorMonitoring(metricRegistry, this::updateJugglerStatus);

            if (needLogbroker) {
                initializeLogbrokerReader(baseEssConfig, metricRegistry);
            }
            initialize();
            isInitialized = true;
        }
    }

    private void initializeLogbrokerReader(BaseEssConfig baseEssConfig, MetricRegistry metricRegistry) {

        Class<? extends BaseLogicObject> logicObjectClass = baseEssConfig.getLogicObject();
        EssCommonUtils essCommonUtils = this.essLogicProcessorContext.getEssCommonUtils();
        String logbrokerTopic = essCommonUtils.getAbsoluteTopicPath(baseEssConfig.getTopic());
        int rowThreshold = baseEssConfig.getRowsThreshold();
        Duration timeToReadThreshold = baseEssConfig.getTimeToReadThreshold();

        Integer shard = getShard();

        LogicProcessorLogbrokerCommonConfig essLogbrokerConfig =
                this.essLogicProcessorContext.getLogicProcessorLogbrokerCommonConfig();

        LogicProcessorLogbrokerProperties logicProcessorLogbrokerProperties =
                new LogicProcessorLogbrokerProperties(essLogicProcessorContext.getLogicProcessorLogbrokerCommonConfig(),
                        logbrokerTopic, singletonList(shard));

        Supplier<SyncConsumer> syncConsumerSupplier = essLogicProcessorContext.getLogbrokerClientFactory()
                .createConsumerSupplier(logicProcessorLogbrokerProperties);

        this.logbrokerReader = new RetryingLogbrokerBatchReader<>(
                () -> new LogicObjectLogbrokerReader<>(syncConsumerSupplier, false,
                        logicObjectClass, rowThreshold, timeToReadThreshold, metricRegistry, true),
                essLogbrokerConfig.getRetries());
    }

    @Override
    public void execute() {
        // если logic-processor отключен - job все равно будет запускаться, но обрабатывать ничего не будет
        // чтобы непрерывно не крутиться в цикле и не засорять лог - job засыпает на время, равное времени чтения из
        // lb(примерно тому, которое он бы работал)
        Set<String> disabledProcessors = essLogicProcessorContext.getDisabledLogicProcessorsProp().getOrDefault(emptySet());
        if (disabledProcessors.contains(logicProcessName)) {
            ThreadUtils.sleep(baseEssConfig.getTimeToReadThreshold());
            finish();
            return;
        }
        initializeInternal();
        try {
            logbrokerReader.fetchEvents(processor);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new LogicProcessorException(e);
        }
    }


    /**
     * It's used only for debug running.
     */
    @SuppressWarnings("unchecked")
    public void processBaseObjects(List<BaseLogicObject> baseLogicObjects) {
        initializeInternal(false);
        process(baseLogicObjects.stream().map(baseLogicObject -> (T) baseLogicObject).collect(Collectors.toList()));
    }

    protected void finishInitialized() {
    }

    @Override
    public void finish() {
        if (solomonRegistryLabels != null) {
            SOLOMON_REGISTRY.removeSubRegistry(solomonRegistryLabels);
        }
        if (logbrokerReader != null) {
            try {
                this.logbrokerReader.close();
            } catch (LogbrokerReaderCloseException ex) {
                logger.error("Error while closing logbrokerReader", ex);
            }
        }
        finishInitialized();
        isInitialized = false;
    }

    private class Processor implements Interrupts.InterruptibleConsumer<List<LogicObjectListWithInfo<T>>> {
        @Override
        public void accept(List<LogicObjectListWithInfo<T>> values) {
            AtomicLong maxTimestamp = new AtomicLong(0);
            long maxPossibleTimestamp = System.currentTimeMillis() / 1000 + 3660;
            List<T> logicObjects = values.stream()
                    .peek(logicObjectWithSystemInfo -> {
                                long currentTimestamp =
                                        logicObjectWithSystemInfo.getUtcTimestamp();
                                if (maxTimestamp.get() < currentTimestamp
                                        && currentTimestamp <= maxPossibleTimestamp) {
                                    maxTimestamp.set(currentTimestamp);
                                }
                            }
                    )
                    .filter(logicObjectWithSystemInfo -> !logicObjectWithSystemInfo.getIsPingObject())
                    .flatMap(logicObjectWithSystemInfo -> logicObjectWithSystemInfo.getLogicObjectsList().stream())
                    .collect(Collectors.toList());

            if (!logicObjects.isEmpty()) {
                processLogicObjects(logicObjects);
            }

            updateMetrics(maxTimestamp.get(), logicObjects.size());
            setMaxTimestamp(maxTimestamp.get());
        }

        private void processLogicObjects(List<T> logicObjects) {
            List<T> objectsToProcess;
            try (var ignore = Trace.current().profile("filter_blacklisted_objects")) {
                objectsToProcess = filterBlacklistedObjects(logicObjects);
            }
            if (!objectsToProcess.isEmpty()) {
                process(objectsToProcess);
            }
        }

        private List<T> filterBlacklistedObjects(List<T> logicObjects) {
            Map<Boolean, List<T>> assortedLogicObjects = logicObjects.stream()
                    .collect(Collectors.partitioningBy(obj -> !blacklist.matches(logicProcessName, obj)));
            List<T> blacklistedObjects = assortedLogicObjects.get(false);
            if (!blacklistedObjects.isEmpty()) {
                String skippedObjects = StreamEx.of(blacklistedObjects)
                        .map(JsonUtils::toJson)
                        .joining(", ");
                logger.info("Some logic objects will be skipped by logic processor {} on shard {}: {}",
                        logicProcessName, getShard(), skippedObjects);
            }

            logicProcessorMonitoring.addBlacklistedObjectCount(blacklistedObjects.size());
            return assortedLogicObjects.get(true);
        }

        private void updateMetrics(long maxTimestamp, int logicObjectsCount) {
            logicProcessorMonitoring.setMaxTimestamp(maxTimestamp);
            logicProcessorMonitoring.addLogicObjectsCount(logicObjectsCount);
        }
    }

    private void setMaxTimestamp(Long maxTimestamp) {
        this.maxTimestamp.set(maxTimestamp);
    }

    private void updateJugglerStatus(long binlogDelay) {
        if (binlogDelay > criticalEssProcessTimeSec) {
            setJugglerStatus(JugglerStatus.CRIT, String.format(
                    "Logic processor %s on shard %d exceeded the critical run time of all ess chain: critical time = " +
                            "%d s but process time = %d s",
                    logicProcessName, getShard(), criticalEssProcessTimeSec, binlogDelay));
        } else {
            setJugglerStatus(JugglerStatus.OK, "OK");
        }
    }

    public AtomicLong getMaxTimestamp() {
        return maxTimestamp;
    }
}
