package ru.yandex.direct.logicprocessor.common;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonProperty;
import kotlin.Metadata;
import kotlin.jvm.internal.Reflection;
import kotlin.reflect.KClass;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.Nullable;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.essblacklist.model.EssBlacklistItem;
import ru.yandex.direct.ess.common.models.BaseLogicObject;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Arrays.asList;
import static org.apache.logging.log4j.core.util.ReflectionUtil.getFieldValue;

@ParametersAreNonnullByDefault
public class BlacklistMatcher<T extends BaseLogicObject> {
    private static Logger logger = LoggerFactory.getLogger(BlacklistMatcher.class);
    private static Map<Class<? extends BaseLogicObject>, List<Field>> logicObjectTypeFields = new ConcurrentHashMap<>();

    private final List<Specification> specs;

    private BlacklistMatcher(List<Specification> specs) {
        this.specs = specs;
    }

    public static <L extends BaseLogicObject> BlacklistMatcher<L> create(List<EssBlacklistItem> blacklist,
                                                                         Class<L> logicObjectType) {
        List<Specification> specs = StreamEx.of(blacklist)
                .mapToEntry(item -> safeDeserializeLogicObject(logicObjectType, item.getFilterSpec()))
                .nonNullValues()
                .mapKeyValue((item, lObj) -> {
                    var fields = getSignificantFields(item.getFilterSpec(), logicObjectType);
                    if (fields.isEmpty()) {
                        return null;
                    }

                    var fieldValues = new HashMap<Field, Object>();
                    fields.forEach(f -> fieldValues.put(f, getFieldValue(f, lObj)));
                    return new Specification(fieldValues);
                })
                .nonNull()
                .toList();

        return new BlacklistMatcher<L>(specs);
    }

    private static <L extends BaseLogicObject> L safeDeserializeLogicObject(Class<L> logicObjectType, String json) {
        try {
            return JsonUtils.fromJson(json, logicObjectType);
        } catch (IllegalArgumentException e) {
            logger.error("Can't deserialize '{}' to {}", json, logicObjectType.getSimpleName());
            return null;
        }
    }

    private static <L extends BaseLogicObject> List<Field> getSignificantFields(String jsonStr,
                                                                                Class<L> logicObjectType) {
        List<Field> fields = getAllFields(logicObjectType);
        var json = new JSONObject(jsonStr);

        var result = StreamEx.of(json.keySet())
                .map(key -> fields.stream()
                        .filter(f -> getJsonPropertyName(f).equals(key))
                        .findAny()
                        .orElse(null))
                .nonNull()
                .toList();
        return result;
    }

    private static String getJsonPropertyName(Field field) {
        JsonProperty annotation = field.getAnnotation(JsonProperty.class);
        if (annotation == null) {
            var kclass = getKotlinClass(field.getDeclaringClass());
            if (kclass != null) {
                annotation = getJsonPropertyFromKotlinConstructor(kclass, field.getName());
            }
        }
        return (annotation == null || annotation.value().equals(JsonProperty.USE_DEFAULT_NAME)) ?
                field.getName() : annotation.value();
    }

    @Nullable
    private static KClass<?> getKotlinClass(Class<?> klass) {
        if (klass.getAnnotation(Metadata.class) != null) {
            return Reflection.getOrCreateKotlinClass(klass);
        } else {
            return null;
        }
    }

    @Nullable
    private static JsonProperty getJsonPropertyFromKotlinConstructor(KClass<?> kclass, String fieldName) {
        return StreamEx.of(kclass.getConstructors())
                .flatMap(constructor -> constructor.getParameters().stream())
                .filter(param -> fieldName.equals(param.getName()))
                .flatMap(param -> param.getAnnotations().stream())
                .select(JsonProperty.class)
                .findFirst().orElse(null);
    }

    private static <L extends BaseLogicObject> List<Field> getAllFields(Class<L> logicObjectType) {
        logicObjectTypeFields.computeIfAbsent(logicObjectType, type -> {
            var fields = new ArrayList<Field>();
            Class<?> currentCls = type;

            while (currentCls != BaseLogicObject.class) {
                fields.addAll(asList(currentCls.getDeclaredFields()));
                currentCls = currentCls.getSuperclass();
            }

            return StreamEx.of(fields)
                    .filter(field -> !Modifier.isStatic(field.getModifiers()))
                    .toList();
        });

        return logicObjectTypeFields.get(logicObjectType);
    }

    public boolean matches(T logicObject) {
        return specs.stream().anyMatch(spec -> spec.isSatisfiedBy(logicObject));
    }

    private static class Specification {
        private final Map<Field, Object> expectedFieldValues;

        Specification(Map<Field, Object> expectedFieldValues) {
            this.expectedFieldValues = expectedFieldValues;
        }

        public boolean isSatisfiedBy(Object logicObject) {
            return EntryStream.of(expectedFieldValues).allMatch(
                    (field, expectedValue) -> Objects.equals(expectedValue, getFieldValue(field, logicObject))
            );
        }
    }
}
