package ru.yandex.direct.logicprocessor.common

import com.google.common.cache.CacheBuilder
import com.google.common.cache.CacheLoader
import com.google.common.util.concurrent.ExecutionError
import com.google.common.util.concurrent.UncheckedExecutionException
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.essblacklist.EssLogicObjectsBlacklistRepository
import ru.yandex.direct.ess.common.models.BaseLogicObject
import java.util.concurrent.ExecutionException
import java.util.concurrent.TimeUnit
import javax.annotation.ParametersAreNonnullByDefault

@Component
@ParametersAreNonnullByDefault
open class EssLogicObjectsBlacklist @Autowired constructor(
    private val essLogicObjectsBlacklistRepository: EssLogicObjectsBlacklistRepository) {

    private val logger = LoggerFactory.getLogger(EssLogicObjectsBlacklist::class.java)

    private val blacklistCache = CacheBuilder.newBuilder()
        .expireAfterWrite(BLACK_LIST_ITEMS_REFRESH_INTERVAL_SEC.toLong(), TimeUnit.SECONDS)
        .build(
            object : CacheLoader<CacheKey, BlacklistMatcher<*>>() {
                override fun load(key: CacheKey): BlacklistMatcher<*> {
                    val blacklistItems = essLogicObjectsBlacklistRepository
                        .getItemsForLogicProcess(key.logicProcessName)
                    return BlacklistMatcher.create(blacklistItems, key.logicObjectClass)
                }
            })

    fun <T : BaseLogicObject> matches(logicProcessName: String,
                                      logicObject: T): Boolean =
        try {
            val matcher = blacklistCache[CacheKey(logicProcessName, logicObject.javaClass)] as BlacklistMatcher<T>
            matcher.matches(logicObject)
            // обрабатываем exception-ы, которые может кинуть метод LoadingCache.get
        } catch (e: ExecutionException) {
            logger.error("ExecutionException while loading blackList for logicProcess $logicProcessName", e)
            false
        } catch (e: UncheckedExecutionException) {
            logger.error("UncheckedExecutionException while loading blackList for logicProcess $logicProcessName", e)
            false
        } catch (e: ExecutionError) {
            logger.error("ExecutionError while loading blackList for logicProcess $logicProcessName", e)
            false
        }

    private data class CacheKey(val logicProcessName: String, val logicObjectClass: Class<out BaseLogicObject?>)

    companion object {
        private const val BLACK_LIST_ITEMS_REFRESH_INTERVAL_SEC = 60
    }

}
