package ru.yandex.direct.logicprocessor.common;

import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;

import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

class LogicProcessorMonitoring {

    private static final String BINLOG_DELAY_LABEL = "binlog_delay";
    private static final String LOGIC_OBJECTS_COUNT_LABEL = "logic_objects_count";
    private static final String BLACKLISTED_OBJECTS_COUNT_LABEL = "blacklisted_objects_count";
    private final MetricRegistry metricRegistry;
    private final AtomicLong binlogDelay = new AtomicLong();
    private final Counter logicObjectsCount;
    private final Rate blacklistedObjectsCount;
    private final Consumer<Long> jugglerConsumer;
    private boolean binlogDelayGaugeInit = false;

    LogicProcessorMonitoring(MetricRegistry metricRegistry, Consumer<Long> jugglerConsumer) {
        this.metricRegistry = metricRegistry;
        this.logicObjectsCount = metricRegistry.counter(LOGIC_OBJECTS_COUNT_LABEL);
        this.blacklistedObjectsCount = metricRegistry.rate(BLACKLISTED_OBJECTS_COUNT_LABEL);
        this.jugglerConsumer = jugglerConsumer;
    }

    void addLogicObjectsCount(int newLogicObjectsCount) {
        logicObjectsCount.add(newLogicObjectsCount);
    }

    void addBlacklistedObjectCount(int newBlacklistedObjectsCount) {
        blacklistedObjectsCount.add(newBlacklistedObjectsCount);
    }

    void setMaxTimestamp(long maxTimestamp) {
        binlogDelay.set(maxTimestamp);
        if (!binlogDelayGaugeInit) {
            metricRegistry.lazyGaugeInt64(
                    BINLOG_DELAY_LABEL,
                    this::calcDelay
            );
            binlogDelayGaugeInit = true;
        }
        jugglerConsumer.accept(calcDelay());
    }

    private long calcDelay() {
        long now = System.currentTimeMillis() / 1000;
        return now - binlogDelay.get();
    }
}
