package ru.yandex.direct.logicprocessor.configuration;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;

import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;
import org.springframework.scheduling.TaskScheduler;

import ru.yandex.direct.bstransport.yt.configuration.BsTransportYtConfiguration;
import ru.yandex.direct.common.configuration.UacYdbConfiguration;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.configuration.CoreConfiguration;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.CampaignModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.ObjectWithEnabledEssModeration;
import ru.yandex.direct.core.entity.uac.grut.GrutContext;
import ru.yandex.direct.core.entity.uac.grut.ThreadLocalGrutContext;
import ru.yandex.direct.core.grut.api.GrutApiProperties;
import ru.yandex.direct.core.grut.api.GrutReplicationApiProperties;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.ess.common.circuits.moderation.ModerationDeletionObjectType;
import ru.yandex.direct.ess.common.configuration.EssCommonConfiguration;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerProperties;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerPropertiesImpl;
import ru.yandex.direct.grid.core.configuration.GridCoreConfiguration;
import ru.yandex.direct.logicprocessor.common.LogicProcessorLogbrokerCommonConfig;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestFilter;
import ru.yandex.direct.logicprocessor.processors.moderation.TestEnvModerationRequestFilter;
import ru.yandex.direct.logicprocessor.processors.moderation.VoidModerationRequestFilter;
import ru.yandex.direct.logicprocessor.processors.moderation.special.deletion.ModerationDeletionRequest;
import ru.yandex.direct.logicprocessor.processors.moderation.special.flags.ModerationFlagsRequest;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmIntegrationImpl;
import ru.yandex.direct.tvm.TvmIntegrationStub;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytcore.spring.YtCoreConfiguration;
import ru.yandex.grut.client.GrutClient;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.compression.CompressionCodec;

import static ru.yandex.direct.common.db.PpcPropertyNames.LIST_OF_OBJECTS_WITH_ENABLED_ESS_MODERATION;
import static ru.yandex.direct.config.EssentialConfiguration.CONFIG_SCHEDULER_BEAN_NAME;
import static ru.yandex.direct.ess.common.utils.EssCommonUtils.getAbsoluteConsumerPath;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Configuration
@Import({
        EssCommonConfiguration.class,
        CoreConfiguration.class,
        GridCoreConfiguration.class,
        YtCoreConfiguration.class,
        BsTransportYtConfiguration.class,
        UacYdbConfiguration.class,
})
@ComponentScan(
        basePackages = {
                "ru.yandex.direct.logicprocessor"
        },
        excludeFilters = {
                @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION),
        }
)
public class EssLogicProcessorConfiguration {
    public static final String MODERATION_TVM_INTEGRATION_BEAN_NAME = "moderationTvmIntegration";
    public static final String MODERATION_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME =
            "moderationLogbrokerCredentialsSupplier";
    public static final String MODERATION_LOGBROKER_CLIENT_FACTORY_BEAN_NAME = "moderationLogbrokerClientFactory";

    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_DEFAULT_REQUEST =
            "moderation_logbroker_producers_properties_default_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_ADGROUP_REQUEST =
            "moderation_logbroker_producers_properties_adgroup_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CPM_VIDEO_REQUEST =
            "moderation_logbroker_producers_properties_cpm_video_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_AD_IMAGE_REQUEST =
            "moderation_logbroker_producers_properties_ad_image_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CPM_GEO_PIN_REQUEST =
            "moderation_logbroker_producers_properties_cpm_geo_pin_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_VERDICT_REQUEST =
            "moderation_logbroker_producers_properties_verdict_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_TURBOLANDING_REQUEST =
            "moderation_logbroker_producers_properties_turbolanding_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SITELINKS_REQUEST =
            "moderation_logbroker_producers_properties_sitelinks_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_DISPLAYHREFS_REQUEST =
            "moderation_logbroker_producers_properties_displayhrefs_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_LOGO_REQUEST =
            "moderation_logbroker_producers_properties_banner_logo_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_VCARD_REQUEST =
            "moderation_logbroker_producers_properties_banner_vcard_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_VIDEO_ADDITION_REQUEST =
            "moderation_logbroker_producers_properties_banner_video_addition_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_BUTTON_REQUEST =
            "moderation_logbroker_producers_properties_banner_button_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_MULTICARD_REQUEST =
            "moderation_logbroker_producers_properties_banner_multicard_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_PROMO_EXTENSION_REQUEST =
            "moderation_logbroker_producers_properties_promo_extension_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNERSTORAGE_CREATIVE_REQUEST =
            "moderation_logbroker_producers_properties_bannerstorage_creative_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_IMAGE_REQUEST =
            "moderation_logbroker_producers_properties_image_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_HTML5_REQUEST =
            "moderation_logbroker_producers_properties_html5_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CANVAS_REQUEST =
            "moderation_logbroker_producers_properties_canvas_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_INTERNAL_BANNER_REQUEST =
            "moderation_logbroker_producers_properties_internal_banner_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_TEXT_REQUEST =
            "moderation_logbroker_producers_properties_text_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_MOBILE_CONTENT_ICON_REQUEST =
            "moderation_logbroker_producers_properties_mobile_content_icon_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_DELETION_REQUEST =
            "moderation_logbroker_producers_properties_special_deletion_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_CAMPAIGN_COPY_REQUEST =
            "moderation_logbroker_producers_properties_special_campaign_copy_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_ARCHIVING_REQUEST =
            "moderation_logbroker_producers_properties_special_archiving_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CALLOUT_REQUEST =
            "moderation_logbroker_producers_properties_callout_request";
    public static final String MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CAMPAIGN_REQUEST =
            "moderation_logbroker_producers_properties_campaign_request";

    @Bean
    public LogicProcessorLogbrokerCommonConfig essLogbrokerConfig(
            @Value("${ess.logbroker.logbroker_prefix}") String logbrokerPrefix,
            @Value("${ess.logbroker.host}") String logbrokerHost,
            @Value("${ess.logbroker.consumer_name}") String consumerName,
            @Value("${ess.logbroker.read_timeout}") int logbrokerReadTimeoutSec,
            @Value("${ess.logbroker.init_timeout}") int logbrokerInitTimeoutSec,
            @Value("${ess.logbroker.retries}") int retries,
            @Value("${ess.logic_processors.logbroker.read_buffer_size}") int readBufferSize,
            @Value("${ess.logic_processors.logbroker.max_count}") int maxCount
    ) {
        String absoluteConsumerPath = getAbsoluteConsumerPath(logbrokerPrefix, consumerName);
        return new LogicProcessorLogbrokerCommonConfig.Builder()
                .setHost(logbrokerHost)
                .setConsumerName(absoluteConsumerPath)
                .setReadDataTimeoutSec(logbrokerReadTimeoutSec)
                .setInitTimeoutSec(logbrokerInitTimeoutSec)
                .setRetries(retries)
                .setReadBufferSize(readBufferSize)
                .setMaxCount(maxCount)
                .build();

    }

    @Bean(name = MODERATION_TVM_INTEGRATION_BEAN_NAME)
    public TvmIntegration moderationTvmIntegration(DirectConfig directConfig,
                                                   @Qualifier(CONFIG_SCHEDULER_BEAN_NAME) TaskScheduler liveConfigChangeTaskScheduler) {
        boolean enabled = directConfig.findBoolean("moderation_service.tvm.enabled").orElse(true);
        if (enabled) {
            return TvmIntegrationImpl.create(directConfig.getBranch("moderation_service"),
                    liveConfigChangeTaskScheduler);
        } else {
            return new TvmIntegrationStub();
        }
    }

    @Bean(name = MODERATION_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME)
    public Supplier<Credentials> getModerationLogbrokerCredentialsSupplier(DirectConfig directConfig,
                                                                           @Qualifier(MODERATION_TVM_INTEGRATION_BEAN_NAME) TvmIntegration tvmIntegration) {
        TvmService tvmService = TvmService.fromStringStrict(directConfig.getString("moderation_service.logbroker" +
                ".tvm_service_name"));
        return () -> {
            String serviceTicket = tvmIntegration.getTicket(tvmService);
            return Credentials.tvm(serviceTicket);
        };
    }

    @Bean(name = MODERATION_LOGBROKER_CLIENT_FACTORY_BEAN_NAME)
    public LogbrokerClientFactoryFacade moderationLogbrokerClientCreatorFactory(
            @Qualifier(MODERATION_TVM_LOGBROKER_CREDENTIALS_SUPPLIER_BEAN_NAME) Supplier<Credentials> logbrokerCredentialsSupplier) {
        return new LogbrokerClientFactoryFacade(logbrokerCredentialsSupplier);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_ADGROUP_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesAdGroupRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.adgroup_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_DEFAULT_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesDefaultRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.banner_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CPM_VIDEO_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCpmVideoRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.cpm_video_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_AD_IMAGE_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesAdImageRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.ad_image_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CPM_GEO_PIN_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCpmGeoPinRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.cpm_geo_pin_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_TURBOLANDING_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesTurbolandingRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.turbolanding_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_DISPLAYHREFS_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesDisplayhrefsRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.displayhrefs_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_LOGO_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesBannerLogoRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.banner_logo_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_BUTTON_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesBannerButtonRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.banner_button_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_VCARD_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesBannerVcardRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.banner_vcard_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_VIDEO_ADDITION_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesBannerVideoAdditionRequest(
            DirectConfig conf) {
        DirectConfig moderationConf =
                conf.getBranch("moderation_service.logbroker.banner_video_addition_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNER_MULTICARD_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesBannerMulticardRequest(
            DirectConfig conf) {
        DirectConfig moderationConf =
                conf.getBranch("moderation_service.logbroker.banner_multicard_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_PROMO_EXTENSION_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesPromoExtensionRequest(
            DirectConfig conf) {
        DirectConfig moderationConf =
                conf.getBranch("moderation_service.logbroker.promo_extension_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_BANNERSTORAGE_CREATIVE_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesBannerstorageCreativeRequest(
            DirectConfig conf) {
        DirectConfig cfg = conf.getBranch("moderation_service.logbroker.bannerstorage_creative_moderation_request");
        return getModerationLogbrokerProducersProperties(cfg);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SITELINKS_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesSitelinksRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.sitelinks_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_IMAGE_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesImagesRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.images_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_HTML5_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesHtml5Request(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.html5_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CANVAS_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCanvasRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.canvas_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_VERDICT_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesVerdictRequest(
            DirectConfig conf) {
        DirectConfig verdictConf = conf.getBranch("moderation_service.logbroker.banner_moderation_verdict");
        return getModerationLogbrokerProducersProperties(verdictConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_INTERNAL_BANNER_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesInternalBannerRequest(
            DirectConfig conf) {
        DirectConfig internalBannerConf =
                conf.getBranch("moderation_service.logbroker.internal_banner_moderation_request");
        return getModerationLogbrokerProducersProperties(internalBannerConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_TEXT_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesTextRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.text_banner_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_MOBILE_CONTENT_ICON_REQUEST)
    @Primary
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesMobileContentIconRequest(
            DirectConfig conf) {
        DirectConfig moderationConf = conf.getBranch("moderation_service.logbroker.mobile_content_icon_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_DELETION_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesSpecialDeletionRequest(
            DirectConfig conf) {
        DirectConfig moderationConf =
                conf.getBranch("moderation_service.logbroker.special_deletion_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_CAMPAIGN_COPY_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesSpecialCampaignCopyRequest(
            DirectConfig conf) {
        DirectConfig moderationConf =
                conf.getBranch("moderation_service.logbroker.special_campaign_copy_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_SPECIAL_ARCHIVING_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesSpecialArchivingRequest(
            DirectConfig conf) {
        DirectConfig moderationConf =
                conf.getBranch("moderation_service.logbroker.special_archiving_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CALLOUT_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCalloutRequest(
            DirectConfig conf) {
        DirectConfig moderationConf =
                conf.getBranch("moderation_service.logbroker.callout_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    @Bean(name = MODERATION_LOGBROKER_PRODUCERS_PROPERTIES_CAMPAIGN_REQUEST)
    public List<LogbrokerProducerProperties> moderationLogbrokerProducersPropertiesCampaignRequest(
            DirectConfig conf) {
        DirectConfig moderationConf =
                conf.getBranch("moderation_service.logbroker.campaign_moderation_request");
        return getModerationLogbrokerProducersProperties(moderationConf);
    }

    private List<LogbrokerProducerProperties> getModerationLogbrokerProducersProperties(DirectConfig conf) {
        CompressionCodec compressionCodec =
                CompressionCodec.valueOf(conf.findString("compression_codec")
                        .orElse("gzip").toUpperCase());

        return mapList(conf.getIntList("group"), group ->
                LogbrokerProducerPropertiesImpl.newBuilder()
                        .setHost(conf.getString("host"))
                        .setWriteTopic(conf.getString("topic"))
                        .setTimeoutSec(conf.getLong("data_timeout"))
                        .setRetries(conf.getInt("retries"))
                        .setGroup(group)
                        .setCompressionCodec(compressionCodec)
                        .build());
    }

    @Bean
    public ModerationRequestFilter moderationRequestFilter(
            EnvironmentType environmentType,
            PpcPropertiesSupport ppcPropertiesSupport) {
        PpcProperty<String> enabledModerationObjects =
                ppcPropertiesSupport.get(LIST_OF_OBJECTS_WITH_ENABLED_ESS_MODERATION,
                        Duration.ofSeconds(15));
        if (!environmentType.isBeta() && !environmentType.isTesting()) {
            return new VoidModerationRequestFilter();
        }

        Predicate<ModerationRequest<?>> commonPredicate = moderationRequest -> {

            List<ObjectWithEnabledEssModeration> objectsWithEnabledModeration =
                    JsonUtils.fromJson(enabledModerationObjects.getOrDefault("[]"),
                            new TypeReference<>() {
                            });

            Set<Long> campaignIds = listToSet(objectsWithEnabledModeration,
                    ObjectWithEnabledEssModeration::getCampaignId);
            Set<Long> bannerIds = listToSet(objectsWithEnabledModeration, ObjectWithEnabledEssModeration::getBannerId);

            if (moderationRequest.getMeta() instanceof BannerModerationMeta) {
                BannerModerationMeta meta = (BannerModerationMeta) moderationRequest.getMeta();
                return campaignIds.contains(meta.getCampaignId()) || bannerIds.contains(meta.getBannerId());
            }
            if (moderationRequest.getMeta() instanceof CampaignModerationMeta) {
                CampaignModerationMeta meta = (CampaignModerationMeta) moderationRequest.getMeta();
                return campaignIds.contains(meta.getCampaignId());
            }
            return true;
        };

        Predicate<ModerationFlagsRequest> flagsPredicate = moderationRequest -> {

            List<ObjectWithEnabledEssModeration> objectsWithEnabledModeration =
                    JsonUtils.fromJson(enabledModerationObjects.getOrDefault("[]"),
                            new TypeReference<>() {
                            });

            Set<Long> campaignIds =
                    listToSet(objectsWithEnabledModeration, ObjectWithEnabledEssModeration::getCampaignId);
            Set<Long> bannerIds =
                    listToSet(objectsWithEnabledModeration, ObjectWithEnabledEssModeration::getBannerId);

            return campaignIds.contains(moderationRequest.getCampaignId()) ||
                    bannerIds.contains(moderationRequest.getBannerId());
        };

        Predicate<ModerationDeletionRequest> deletionPredicate = moderationRequest -> {

            List<ObjectWithEnabledEssModeration> objectsWithEnabledModeration =
                    JsonUtils.fromJson(enabledModerationObjects.getOrDefault("[]"),
                            new TypeReference<>() {
                            });

            Set<Long> bannerIds = listToSet(objectsWithEnabledModeration, ObjectWithEnabledEssModeration::getBannerId);

            if (moderationRequest.getType() == ModerationDeletionObjectType.ADGROUP) {
                return true;
            }
            return bannerIds.contains(moderationRequest.getId());
        };

        return new TestEnvModerationRequestFilter(
                Map.of(
                        ModerationRequest.class, commonPredicate,
                        ModerationFlagsRequest.class, flagsPredicate,
                        ModerationDeletionRequest.class, deletionPredicate
                ));
    }


    // Он уже определяется в JobsEssentialConfiguration, но при этом не проходит тест
    // EssLogicProcessorConfigurationTest,
    // так как он собирает только конфигурацию EssLogicProcessorConfiguration
    // стоит перенести этот тест в jobs, но как быстрофикс добавлю тут определение бина
    @Bean
    public GrutContext grutContext(GrutClient grutClient) {
        return new ThreadLocalGrutContext(grutClient);
    }

    @Bean
    @Lazy
    public GrutApiProperties grutApiProperties(PpcPropertiesSupport ppcPropertiesSupport) {
        return new GrutReplicationApiProperties(ppcPropertiesSupport);
    }
}
