package ru.yandex.direct.logicprocessor.processors.aggregatedstatuses;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.aggregatedstatuses.AggregatedStatusObjectType;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesService;
import ru.yandex.direct.core.aggregatedstatuses.ChangesHolder;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RecalculationDepthEnum;
import ru.yandex.direct.core.aggregatedstatuses.service.AggregatedStatusesResyncQueueService;
import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedStatusQueueEntityType;
import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedStatusResyncQueueEntity;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.aggregatedstatuses.AggregatedStatusesConfig;
import ru.yandex.direct.ess.logicobjects.aggregatedstatuses.AggregatedStatusEventObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.monlib.metrics.primitives.Counter;

import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.adChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.adgroupChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.bidBaseChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.calloutChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.campaignChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.keywordChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.perfCreativeChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.promoExtensionChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.restrictedGeoChangesEvent;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.retargetingChangesEvent;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0_UNSTABLE;
import static ru.yandex.direct.logicprocessor.processors.aggregatedstatuses.AggregatedStatusEventObjectConverter.getEventObjectsFromResyncQueueEntities;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_0_UNSTABLE}
)
@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 10),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_0_UNSTABLE}
)
@EssLogicProcessor(AggregatedStatusesConfig.class)
public class AggregatedStatusesProcessor extends BaseLogicProcessor<AggregatedStatusEventObject> {

    public static final int CHUNK_SIZE = 50_000;

    public static final int MAX_DELAY_FOR_PROCESSING_RESYNC_ENTITY_QUEUE = 10;

    private final org.slf4j.Logger logger = LoggerFactory.getLogger(AggregatedStatusesProcessor.class);
    private final AggregatedStatusesService aggregatedStatusesService;
    private final AggregatedStatusesResyncQueueService aggregatedStatusesResyncQueueService;
    private final PpcProperty<Set<Long>> cidsToSkipProp;
    private Counter failedChunksCounter;

    @Autowired
    public AggregatedStatusesProcessor(EssLogicProcessorContext essLogicProcessorContext,
                                       AggregatedStatusesService aggregatedStatusesService,
                                       AggregatedStatusesResyncQueueService aggregatedStatusesResyncQueueService,
                                       PpcPropertiesSupport ppcPropertiesSupport) {
        super(essLogicProcessorContext);
        this.aggregatedStatusesService = aggregatedStatusesService;
        this.aggregatedStatusesResyncQueueService = aggregatedStatusesResyncQueueService;
        this.cidsToSkipProp = ppcPropertiesSupport.get(PpcPropertyNames.CIDS_TO_SKIP_IN_STATUSES_PROCESSOR, Duration.ofMinutes(3));
    }

    @Override
    protected void initialize() {
        this.failedChunksCounter = getMetricRegistry().counter("aggregated_statuses_failed_chunks");
    }

    @Override
    public void process(List<AggregatedStatusEventObject> allLogicObjects) {
        int shard = getShard();

        boolean canRecalcStatusesFromResyncQueue = canRecalcStatusesFromResyncQueue();

        var resyncQueueEntities = getResyncQueueEntitiesAndMarkAsProcessing(shard,
                canRecalcStatusesFromResyncQueue);

        var eventObjects = fetchEventObjectsToProcess(allLogicObjects, resyncQueueEntities);
        var cidsToSkip = cidsToSkipProp.getOrDefault(Set.of());
        eventObjects = eventObjects.stream()
                .filter(t -> t.getCampaignId() == null || !cidsToSkip.contains(t.getCampaignId()))
                .collect(Collectors.toList());
        StreamEx.ofSubLists(eventObjects, CHUNK_SIZE)
                .forEach(eventObjectSubList -> processChunk(shard, eventObjectSubList));

        deleteProcessedAggregatedStatusResyncQueueEntities(shard, resyncQueueEntities,
                canRecalcStatusesFromResyncQueue);
    }

    /**
     * Считаем, что если последний timestamp был в течении
     * {@link AggregatedStatusesProcessor#MAX_DELAY_FOR_PROCESSING_RESYNC_ENTITY_QUEUE}
     * aka отставание не критичное, можем пересчитать статус из resync_queue
     */
    private boolean canRecalcStatusesFromResyncQueue() {
        long nowInSec = System.currentTimeMillis() / 1000;
        return nowInSec - getMaxTimestamp().get() < MAX_DELAY_FOR_PROCESSING_RESYNC_ENTITY_QUEUE;
    }

    private List<AggregatedStatusResyncQueueEntity> getResyncQueueEntitiesAndMarkAsProcessing(
            int shard,
            boolean canRecalcStatusesFromResyncQueue) {
        if (canRecalcStatusesFromResyncQueue) {
            logger.debug("start recalculating statuses from resync queue");
            var resyncQueueEntities =
                    aggregatedStatusesResyncQueueService.getResyncQueueEntitiesAndMarkAsProcessing(shard);
            logUnknownEntities(resyncQueueEntities);
            return resyncQueueEntities;
        }
        return Collections.emptyList();
    }

    //в норме записей с неизвестным типом не должно быть
    private void logUnknownEntities(List<AggregatedStatusResyncQueueEntity> resyncQueueEntities) {
        List<Long> unknownEntityIds = filterAndMapList(resyncQueueEntities,
                entity -> AggregatedStatusQueueEntityType.UNKNOWN == entity.getType(),
                AggregatedStatusResyncQueueEntity::getId);
        if (!unknownEntityIds.isEmpty()) {
            logger.error("Got unknown aggregated status queue entity type for ids: {}", unknownEntityIds);
        }
    }

    private List<AggregatedStatusEventObject> fetchEventObjectsToProcess(
            List<AggregatedStatusEventObject> allLogicObjects,
            List<AggregatedStatusResyncQueueEntity> resyncQueueEntities) {
        var eventObjectsFromResyncQueue = getEventObjectsFromResyncQueueEntities(resyncQueueEntities);
        return resyncQueueEntities.isEmpty() ?
                allLogicObjects :
                ListUtils.union(allLogicObjects, eventObjectsFromResyncQueue);
    }

    private void processChunk(int shard, List<AggregatedStatusEventObject> logicObjects) {
        Map<AggregatedStatusObjectType, List<AggregatedStatusEventObject>> changedObjectsByType =
                logicObjects.stream().collect(Collectors.groupingBy(AggregatedStatusEventObject::getType));

        for (AggregatedStatusObjectType type : changedObjectsByType.keySet()) {
            List<AggregatedStatusEventObject> eventObjects = changedObjectsByType.get(type);
            String idsString = eventObjects.stream()
                    .map(o -> String.valueOf(o.getId())).collect(Collectors.joining(","));
            logger.info("processing {} changes for {} with ids: {}", eventObjects.size(), type.name(), idsString);
        }

        var changes = new ChangesHolder(
                changedObjectsByType.getOrDefault(AggregatedStatusObjectType.CAMPAIGN, Collections.emptyList())
                        .stream().map(o -> campaignChangesEvent(o.getId(), o.isDeleted()))
                        .collect(Collectors.toUnmodifiableSet()),
                changedObjectsByType.getOrDefault(AggregatedStatusObjectType.ADGROUP, Collections.emptyList())
                        .stream().map(o -> adgroupChangesEvent(o.getId(), o.getParentId(), o.isDeleted()))
                        .collect(Collectors.toUnmodifiableSet()),
                changedObjectsByType.getOrDefault(AggregatedStatusObjectType.AD, Collections.emptyList())
                        .stream().map(o -> adChangesEvent(o.getId(), o.getParentId(), o.isDeleted()))
                        .collect(Collectors.toUnmodifiableSet()),
                changedObjectsByType.getOrDefault(AggregatedStatusObjectType.KEYWORD, Collections.emptyList())
                        .stream().map(o ->
                        keywordChangesEvent(o.getId(), o.getParentId(), o.getCampaignId(), o.isDeleted()))
                        .collect(Collectors.toUnmodifiableSet()),
                changedObjectsByType.getOrDefault(
                        AggregatedStatusObjectType.BID_BASE, Collections.emptyList())
                        // У объектов relevance match из bids_base есть статус, но удаление оттуда невозможно отличить
                        // от архивации keyword (нету поля type и/или части ключа чтобы посмотреть в bids_arc),
                        // поэтому статусы relevance match не удаляем
                        .stream().map(o -> bidBaseChangesEvent(o.getId(), o.getParentId()))
                        .collect(Collectors.toUnmodifiableSet()),
                changedObjectsByType.getOrDefault(AggregatedStatusObjectType.CALLOUT, Collections.emptyList())
                        .stream().map(o -> calloutChangesEvent(o.getId()))
                        .collect(Collectors.toUnmodifiableSet()),
                changedObjectsByType.getOrDefault(AggregatedStatusObjectType.PERF_CREATIVES, Collections.emptyList())
                        .stream().map(o -> perfCreativeChangesEvent(o.getId()))
                        .collect(Collectors.toUnmodifiableSet()),
                changedObjectsByType.getOrDefault(
                        AggregatedStatusObjectType.RESTRICTED_GEO, Collections.emptyList())
                        .stream().map(o -> restrictedGeoChangesEvent(o.getId()))
                        .collect(Collectors.toUnmodifiableSet()),
                changedObjectsByType.getOrDefault(
                        AggregatedStatusObjectType.RETARGETING, Collections.emptyList())
                        .stream().map(r -> retargetingChangesEvent(r.getId(), r.getParentId(), r.isDeleted()))
                        .collect(Collectors.toUnmodifiableSet()),
                changedObjectsByType.getOrDefault(
                        AggregatedStatusObjectType.PROMO_EXTENSION, Collections.emptyList())
                        .stream().map(r -> promoExtensionChangesEvent(r.getId()))
                        .collect(Collectors.toUnmodifiableSet())
        );

        try {
            aggregatedStatusesService.processChanges(shard, null, changes, RecalculationDepthEnum.ALL);
        } catch (RuntimeException e) {
            logger.error("Aggregated statuses chunk processing failed: ", e);
            var changesHolderLogger = new ChangesHolder.ChangesHolderLogger(changes, logger::error);
            changesHolderLogger.logAllIdStrings();

            failedChunksCounter.inc();
        }
    }

    private void deleteProcessedAggregatedStatusResyncQueueEntities(
            int shard,
            List<AggregatedStatusResyncQueueEntity> resyncQueueEntities,
            boolean canRecalcStatusesFromResyncQueue) {
        if (!canRecalcStatusesFromResyncQueue) {
            return;
        }
        List<Long> resyncIds = mapList(resyncQueueEntities, AggregatedStatusResyncQueueEntity::getId);
        aggregatedStatusesResyncQueueService.deleteProcessedAggregatedStatusResyncQueueEntities(shard, resyncIds);
        logger.debug("finish recalculating statuses from resync queue");
    }
}
