package ru.yandex.direct.logicprocessor.processors.brandliftrecalc;

import java.util.List;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.brandlift.model.BrandLiftRecalcJobParams;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.ess.logicobjects.brandliftrecalc.BrandLiftRecalcObject;
import ru.yandex.direct.utils.FunctionalUtils;

import static com.google.common.collect.Lists.partition;
import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.RECALC_BRAND_LIFT_CAMPAIGNS;
import static ru.yandex.direct.dbutil.SqlUtils.TYPICAL_SELECT_CHUNK_SIZE;

@Service
public class BrandLiftRecalcService {
    private final Logger logger = LoggerFactory.getLogger(BrandLiftRecalcService.class);

    private final DbQueueRepository dbQueueRepository;
    private final CampaignRepository campaignRepository;

    @Autowired
    public BrandLiftRecalcService(
            DbQueueRepository dbQueueRepository,
            CampaignRepository campaignRepository
    ) {
        this.dbQueueRepository = dbQueueRepository;
        this.campaignRepository = campaignRepository;
    }

    void processRecalc(int shard, List<BrandLiftRecalcObject> incomingObjects) {
        for (List<BrandLiftRecalcObject> objects : partition(incomingObjects, TYPICAL_SELECT_CHUNK_SIZE)) {
            List<Long> cids = FunctionalUtils.mapList(objects, BrandLiftRecalcObject::getCid);

            // достаем данные о БЛ для каждой кампании
            var cidsBrandSurveyId = campaignRepository.getBrandSurveyIdsForCampaigns(shard, cids);
            var cidsWithBrandLift = StreamEx.of(cids)
                    .filter(cid -> {
                        var brandSurveyId = cidsBrandSurveyId.get(cid);
                        return brandSurveyId != null && !brandSurveyId.isEmpty();
                    }).toList();
            // кампанию с бренд лифтом ставим в очередь DBQueue
            insertCidToDBQueue(shard, cidsWithBrandLift);
        }
    }

    /*
     * добавление номера кампании в очередь DBQueue для последующего пересчета.
     */
    private void insertCidToDBQueue(int shard, List<Long> cidsToRecalc) {
        if (cidsToRecalc.isEmpty()) {
            logger.info("Empty campaigns list for adding to db_queue");
            return;
        }
        var clientIdsForCids = campaignRepository.getClientIdsForCids(shard, cidsToRecalc);

        for (Long campaignId : cidsToRecalc) {
            BrandLiftRecalcJobParams jobParams = new BrandLiftRecalcJobParams().withCampaignId(campaignId);
            Long jobId = dbQueueRepository.insertJob(
                    shard,
                    RECALC_BRAND_LIFT_CAMPAIGNS,
                    ClientId.fromLong(clientIdsForCids.get(campaignId)),
                    0L, // не знаю на что влияет uid и какой надо ставить. Поэтому поставила 0.
                    jobParams).getId();
            logger.info("Added for BrandLiftRecalc job with id " + jobId);
        }
    }
}
