package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.AdGroup
import ru.yandex.direct.bstransport.yt.repository.adgroup.resources.AdGroupDeleteYtRepository
import ru.yandex.direct.bstransport.yt.repository.adgroup.resources.AdGroupYtRepository
import ru.yandex.direct.bstransport.yt.utils.CaesarIterIdGenerator
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData
import ru.yandex.direct.common.log.service.LogBsExportEssService
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupResourceType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupObject
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler.AdGroupBaseHandler
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler.AdGroupDeleteHandler
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler.AdGroupWithBuilder
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.SupportedCampaignsService
import java.time.Clock
import java.util.IdentityHashMap

@Component
class BsExportAdGroupService(
    private val logBsExportEssService: LogBsExportEssService,
    private val caesarIterIdGenerator: CaesarIterIdGenerator,
    private val bsOrderIdCalculator: BsOrderIdCalculator,
    private val ytRepository: AdGroupYtRepository,
    private val ytDeleteRepository: AdGroupDeleteYtRepository,
    private val clock: Clock = Clock.systemUTC(),
    private val resourceHandlers: List<AdGroupBaseHandler<*>>,
    private val deleteHandler: AdGroupDeleteHandler,
    private val adGroupRepository: AdGroupRepository,
    private val supportedCampaignService: SupportedCampaignsService,
) {

    companion object {
        private const val LOG_TYPE = "ad_group"
    }

    private val resourceHandlersMap: Map<AdGroupResourceType, List<AdGroupBaseHandler<*>>> = resourceHandlers
        .map { it.resourceType() to listOf(it) }
        .toMap()

    fun processAdGroups(shard: Int, logicObjects: List<BsExportAdGroupObject>) {
        val (deletedObjects, updatedObjects) = logicObjects.partition { it.resourceType == AdGroupResourceType.DELETE }
        val iterId = caesarIterIdGenerator.generateCaesarIterId()
        modifyAdGroups(shard, updatedObjects, iterId)
        markAdGroupsAsDeleted(shard, deletedObjects, iterId)
    }

    private fun modifyAdGroups(shard: Int, objects: List<BsExportAdGroupObject>, iterId: Long) {
        if (objects.isEmpty()) {
            return
        }
        val adGroupIdsToLoad = getAdGroupIdsToLoad(shard, objects)
        val adGroups = adGroupRepository.getAdGroups(shard, adGroupIdsToLoad)
        val buildersByAdGroupId = adGroups
            .associate {
                val builder = AdGroup.newBuilder()
                    .setAdGroupId(it.id)
                it.id to AdGroupWithBuilder(it, builder)
            }

        resourceHandlers.forEach { it.handle(shard, buildersByAdGroupId) }

        val campaignIds = adGroups.map { it.campaignId }.distinct()
        val campaignToOrderId = bsOrderIdCalculator.calculateOrderIdIfNotExist(shard, campaignIds)

        val updateTime = clock.instant().epochSecond
        val updatedAdGroups = buildersByAdGroupId.values.asSequence()
            .filter { campaignToOrderId.containsKey(it.adGroup.campaignId) }
            .map {
                it.protoBuilder.orderId = campaignToOrderId[it.adGroup.campaignId]!!
                it.protoBuilder.iterId = iterId
                it.protoBuilder.updateTime = updateTime
                AdGroupExportInfo(it.protoBuilder.build(), it.adGroup.campaignId)
            }
            .toList()
        if (updatedAdGroups.isNotEmpty()) {
            ytRepository.modify(updatedAdGroups.map { it.adGroup })
            logResources(updatedAdGroups)
        }
    }

    private fun markAdGroupsAsDeleted(shard: Int, objects: List<BsExportAdGroupObject>, iterId: Long) {
        if (objects.isEmpty()) {
            return
        }
        val deletedAdGroups = deleteHandler.getDeletedAdGroups(shard, objects)
        val adGroupIdsToCampaignIds = deletedAdGroups
            .distinctBy { it.adGroupId }.associate { it.adGroupId to it.campaignId }
        val campaignToOrderId = bsOrderIdCalculator
            .calculateOrderIdIfNotExist(shard, adGroupIdsToCampaignIds.values.distinct())

        val now = clock.instant().epochSecond

        val resources = deletedAdGroups
            .filter { campaignToOrderId.containsKey(it.campaignId) }
            .map {
                AdGroup.newBuilder()
                    .setIterId(iterId)
                    .setUpdateTime(now)
                    .setDeleteTime(now)
                    .setAdGroupId(it.adGroupId)
                    .setOrderId(campaignToOrderId.getValue(it.campaignId))
                    .build()
            }
            .map {
                AdGroupExportInfo(it, adGroupIdsToCampaignIds.getValue(it.adGroupId))
            }
            .toList()

        if (resources.isNotEmpty()) {
            ytDeleteRepository.modify(resources.map { it.adGroup })
            logResources(resources)
        }
    }

    private fun logResources(adGroups: List<AdGroupExportInfo>) {
        val logsData = adGroups
            .map { info ->
                LogBsExportEssData<AdGroup>()
                    .withPid(info.adGroup.adGroupId)
                    .withCid(info.campaignId)
                    .withOrderId(info.adGroup.orderId)
                    .withData(info.adGroup)
            }
        logBsExportEssService.logData(logsData, LOG_TYPE)
    }

    private fun getAdGroupIdsToLoad(shard: Int, logicObjects: Collection<BsExportAdGroupObject>): List<Long> {
        val adGroupIdsToLoad = logicObjects
            .asSequence()
            .map { it.resourceType to it }
            .map { (type, obj) -> getHandlers(type) to obj }
            .flatMap { (handlers, obj) -> handlers.map { it to obj } }
            .groupByTo(IdentityHashMap(), { it.first }, { it.second })
            .flatMap { (handler, objects) -> handler.getAdGroupIdsToLoad(shard, objects) }
            .distinct()

        val badCpmBannerGroups = adGroupRepository
            .getCpmBannersAdGroupsWithoutCriterionType(shard, adGroupIdsToLoad)
            .toSet()
        val unsupportedCampaignAdGroups = supportedCampaignService
            .getAdGroupIdsWithUnsupportedCampaign(shard, adGroupIdsToLoad)
            .toSet()

        return adGroupIdsToLoad.filterNot {
            it in badCpmBannerGroups || it in unsupportedCampaignAdGroups
        }
    }

    private fun getHandlers(resourceType: AdGroupResourceType): List<AdGroupBaseHandler<*>> =
        if (AdGroupResourceType.ALL == resourceType) {
            resourceHandlers
        } else {
            resourceHandlersMap.getOrDefault(resourceType, listOf())
        }
}
