package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DISALLOW_CAMPAIGN_NAME_LETTERS_RE
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupResourceType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupObject
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.AdGroupResource
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer.BsHrefParametrizingService
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer.ReplacingParams
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.CampaignNameTransliterator
import ru.yandex.direct.multitype.entity.LimitOffset

typealias ClickUrlTail = String

@Component
class AdGroupClickUrlTailHandler(
    private val campaignTypedRepository: CampaignTypedRepository,
    private val bsHrefParametrizingService: BsHrefParametrizingService,
    private val transliterator: CampaignNameTransliterator,
    private val adGroupRepository: AdGroupRepository,
) : AdGroupBaseHandler<ClickUrlTail>() {
    companion object {
        private val logger = LoggerFactory.getLogger(AdGroupClickUrlTailHandler::class.java)
        private val allowedAdGroupTypes = setOf(AdGroupType.PERFORMANCE, AdGroupType.DYNAMIC)
        private val DISALLOWED_CAMPAIGN_NAME_LETTERS = Regex(DISALLOW_CAMPAIGN_NAME_LETTERS_RE)
    }

    override fun getAdGroupIdsToLoad(shard: Int, objects: Collection<BsExportAdGroupObject>): List<Long> {
        val adGroupIdsFromObjects = objects.filter { it.adGroupId != null }.map { it.adGroupId!! }
        val campaignIdsWithoutAdGroupId = objects.filter { it.adGroupId == null }.map { it.campaignId }.toSet()

        return if (campaignIdsWithoutAdGroupId.isNotEmpty()) {
            val selectionCriteria = AdGroupsSelectionCriteria().withCampaignIds(campaignIdsWithoutAdGroupId).withAdGroupTypes(allowedAdGroupTypes)
            adGroupIdsFromObjects + adGroupRepository.getAdGroupIdsBySelectionCriteria(shard, selectionCriteria, LimitOffset.maxLimited())
        } else adGroupIdsFromObjects
    }

    override fun resourceType(): AdGroupResourceType {
        return AdGroupResourceType.CLICK_URL_TAIL
    }

    override fun loadResources(
        shard: Int,
        adGroups: Collection<AdGroup>,
    ): List<AdGroupResource<ClickUrlTail>> {
        val campaigns = getPidToCampaignMap(shard, adGroups)
        logger.info("Going to handle ${adGroups.size} adgroups")

        return adGroups
            .mapNotNull { adGroup ->
                val campaign = campaigns[adGroup.id] ?: return@mapNotNull null
                val clickUrlTail = getClickUrlTail(adGroup, campaign) ?: ""
                AdGroupResource(
                    adGroupId = adGroup.id,
                    campaignId = adGroup.campaignId,
                    resource = clickUrlTail,
                )
            }
    }

    private fun getClickUrlTail(adGroup: AdGroup, campaign: CommonCampaign): String? {
        val urlTemplate: String? = adGroup.trackingParams
        return when {
            adGroup.type !in allowedAdGroupTypes -> null
            urlTemplate.isNullOrEmpty() -> null
            else -> {
                val replacingParams = createReplacingParams(adGroup, campaign)
                bsHrefParametrizingService.parameterize(urlTemplate, replacingParams)
            }
        }
    }

    private fun createReplacingParams(adGroup: AdGroup, campaign: CommonCampaign): ReplacingParams? {
        val campaignName = campaign.name
            ?.replace(DISALLOWED_CAMPAIGN_NAME_LETTERS, "") ?: ""
        var builder = ReplacingParams.builder()
            .withPid(adGroup.id)
            .withCid(adGroup.campaignId)
            .withCampaignType(campaign.type)
            .withCampaignName(campaignName)
            .withCampaignNameLat(transliterator.translit(campaignName))
            .withCampaignCurrency(campaign.currency)
        if (campaign is CampaignWithStrategy) {
            builder = builder
                .withCampaignStrategy(campaign.strategy)
        }
        return builder.build()
    }

    private fun getPidToCampaignMap(shard: Int, adGroups: Collection<AdGroup>): Map<Long, CommonCampaign?> {
        val campaigns = campaignTypedRepository
            .getSafely(
                shard,
                adGroups.map { it.campaignId },
                CommonCampaign::class.java,
            )
            .associateBy { it.id }

        return adGroups
            .associateBy(
                keySelector = { it.id },
                valueTransform = { campaigns[it.campaignId] }
            )
    }

    override fun fillExportObject(
        resource: ClickUrlTail,
        builder: ru.yandex.adv.direct.adgroup.AdGroup.Builder,
    ) {
        builder.clickUrlTail = resource
    }
}
