package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.AdGroup
import ru.yandex.adv.direct.adgroup.AdGroupType
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.product.service.ProductService
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupResourceType
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.AdGroupResource
import ru.yandex.direct.utils.FunctionalUtils.listToSet
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType as CoreAdGroupType

const val DIRECT_ENGINE_ID = 7L

@Component
class AdGroupCommonFieldsHandler(
    private val campaignRepository: CampaignRepository,
    private val productService: ProductService,
) : AdGroupBaseHandler<CommonFields>() {

    override fun resourceType() = AdGroupResourceType.COMMON_FIELDS

    override fun loadResources(shard: Int, adGroups: Collection<ru.yandex.direct.core.entity.adgroup.model.AdGroup>):
        List<AdGroupResource<CommonFields>> {
        val campaignIds: Set<Long> = listToSet(adGroups) { it.campaignId }

        val campaignToEngineId: Map<Long, Long> = obtainEngineIds(shard, campaignIds)
        return adGroups
            .map {
                AdGroupResource(
                    adGroupId = it.id,
                    campaignId = it.campaignId,
                    CommonFields(
                        type = it.type,
                        engineId = campaignToEngineId.getOrDefault(it.campaignId, DIRECT_ENGINE_ID),
                        name = it.name ?: ""
                    )
                )
            }
            .toList()
    }

    override fun fillExportObject(resource: CommonFields, builder: AdGroup.Builder) {
        builder.apply {
            type = AdGroupTypeConverter.toExportType(resource.type).number
            engineId = resource.engineId
            name = resource.name
        }
    }

    private fun obtainEngineIds(shard: Int, campaignIds: Set<Long>): Map<Long, Long> {
        val campaignToProduct: Map<Long, Long?> = campaignRepository.getProductIds(shard, campaignIds)
        return campaignToProduct
            .mapNotNull { entry -> getEngineIdOrNull(entry.value)?.let { value -> Pair(entry.key, value) } }
            .toMap()
    }

    private fun getEngineIdOrNull(productId: Long?): Long? {
        if (productId == null) {
            return null
        }
        return try {
            productService.getProductById(productId).engineId
        } catch (e: NoSuchElementException) {
            null
        }
    }
}

data class CommonFields(
    val type: CoreAdGroupType,
    val engineId: Long,
    val name: String,
)

internal object AdGroupTypeConverter {
    private val coreToExportTypes: Map<CoreAdGroupType, AdGroupType> = mapOf(
        CoreAdGroupType.BASE to AdGroupType.AD_GROUP_TYPE_BASE,
        CoreAdGroupType.DYNAMIC to AdGroupType.AD_GROUP_TYPE_DYNAMIC,
        CoreAdGroupType.MOBILE_CONTENT to AdGroupType.AD_GROUP_TYPE_MOBILE_CONTENT,
        CoreAdGroupType.PERFORMANCE to AdGroupType.AD_GROUP_TYPE_PERFORMANCE,
        CoreAdGroupType.MCBANNER to AdGroupType.AD_GROUP_TYPE_MCBANNER,
        CoreAdGroupType.CPM_BANNER to AdGroupType.AD_GROUP_TYPE_CPM_BANNER,
        CoreAdGroupType.CPM_VIDEO to AdGroupType.AD_GROUP_TYPE_CPM_VIDEO,
        CoreAdGroupType.CPM_OUTDOOR to AdGroupType.AD_GROUP_TYPE_CPM_OUTDOOR,
        CoreAdGroupType.CPM_YNDX_FRONTPAGE to AdGroupType.AD_GROUP_TYPE_CPM_YNDX_FRONTPAGE,
        CoreAdGroupType.CPM_INDOOR to AdGroupType.AD_GROUP_TYPE_CPM_INDOOR,
        CoreAdGroupType.INTERNAL to AdGroupType.AD_GROUP_TYPE_INTERNAL,
        CoreAdGroupType.CONTENT_PROMOTION_VIDEO to AdGroupType.AD_GROUP_TYPE_CONTENT_PROMOTION_VIDEO,
        CoreAdGroupType.CPM_GEOPRODUCT to AdGroupType.AD_GROUP_TYPE_CPM_GEOPRODUCT,
        CoreAdGroupType.CPM_AUDIO to AdGroupType.AD_GROUP_TYPE_CPM_AUDIO,
        CoreAdGroupType.CONTENT_PROMOTION to AdGroupType.AD_GROUP_TYPE_CONTENT_PROMOTION,
        CoreAdGroupType.CPM_PRICE to AdGroupType.AD_GROUP_TYPE_CPM_PRICE,
        CoreAdGroupType.CPM_GEO_PIN to AdGroupType.AD_GROUP_TYPE_CPM_GEO_PIN
    )

    fun toExportType(type: CoreAdGroupType): AdGroupType {
        return coreToExportTypes[type] ?: AdGroupType.AD_GROUP_TYPE_INVALID
    }
}
