package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupResourceType
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.AdGroupResource

typealias MatchPriority = Int

@Component
class AdGroupMatchPriorityHandler(
    private val adgroupRepository: AdGroupRepository,
    private val campaignRepository: CampaignRepository,
) : AdGroupBaseHandler<MatchPriority>() {
    companion object {
        private val logger = LoggerFactory.getLogger(AdGroupMatchPriorityHandler::class.java)
    }

    override fun resourceType(): AdGroupResourceType {
        return AdGroupResourceType.MATCH_PRIORITY
    }

    override fun loadResources(
        shard: Int,
        adGroups: Collection<AdGroup>,
    ): List<AdGroupResource<MatchPriority>> {
        val priorities = adgroupRepository.getAdGroupsPriority(shard, adGroups.map { it.id })
        val campaigns = campaignRepository.getCampaignsTypeMap(shard, adGroups.map { it.campaignId })
        logger.info("Going to handle ${adGroups.size} adgroups")

        return adGroups
            .mapNotNull { adGroup ->
                val campaignType = campaigns[adGroup.campaignId]
                when {
                    adGroup.type == AdGroupType.CPM_YNDX_FRONTPAGE -> {
                        val priority = priorities[adGroup.id]?.toInt()
                        AdGroupResource(
                            adGroupId = adGroup.id,
                            campaignId = adGroup.campaignId,
                            resource = priority ?: 0,
                        )
                    }
                    campaignType == CampaignType.CPM_PRICE && adGroup.type == AdGroupType.CPM_VIDEO -> {
                        AdGroupResource(
                            adGroupId = adGroup.id,
                            campaignId = adGroup.campaignId,
                            resource = 0,
                        )
                    }
                    else -> null
                }
            }
    }

    override fun fillExportObject(
        resource: MatchPriority,
        builder: ru.yandex.adv.direct.adgroup.AdGroup.Builder,
    ) {
        builder.matchPriority = resource
    }
}
