package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.multipliers.Multiplier
import ru.yandex.direct.core.bsexport.repository.BsExportMultipliersRepository
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupResourceType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupObject
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.MultiplierInfo
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.RetargetingConditionInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.AdGroupResource
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.BsExportMultipliersService

typealias Multipliers = List<Multiplier>

@Component
class AdGroupMultipliersHandler(
    private val bsExportMultipliersRepository: BsExportMultipliersRepository,
    private val bsExportMultipliersService: BsExportMultipliersService,
) : AdGroupBaseHandler<Multipliers>() {

    override fun resourceType() = AdGroupResourceType.MULTIPLIERS

    override fun getAdGroupIdsToLoad(shard: Int, objects: Collection<BsExportAdGroupObject>): List<Long> {
        val groupIds = objects.mapNotNullTo(hashSetOf()) { it.adGroupId }
        groupIds += getAdGroupIdsFromMultiplierInfo(shard, objects)
        return groupIds.toList()
    }

    private fun getAdGroupIdsFromMultiplierInfo(shard: Int, objects: Collection<BsExportAdGroupObject>): Set<Long> {
        val retargetingIds = objects
            .mapNotNull {
                (it.additionalInfo as? RetargetingConditionInfo)?.retargetingConditionId
                    ?: (it.additionalInfo as? MultiplierInfo)?.retargetingConditionId
            }
        val multipliersWithRetargetingIds = bsExportMultipliersRepository
            .getMultiplierIdsByRetargetingConditionIds(shard, retargetingIds)
            .toSet()

        val multiplierIds = multipliersWithRetargetingIds + objects
            .mapNotNull { (it.additionalInfo as? MultiplierInfo)?.hierarchicalMultiplierId }
        return bsExportMultipliersRepository
            .getAdGroupIdsByHierarchicalMultiplierIds(shard, multiplierIds)
            .toSet()
    }

    override fun loadResources(shard: Int, adGroups: Collection<AdGroup>): List<AdGroupResource<Multipliers>> {
        val adGroupMultipliers = bsExportMultipliersService
            .getAdGroupMultipliers(shard, adGroups.map { it.id })
        return adGroups.map {
            val multiplier = adGroupMultipliers[it.id] ?: emptyList()
            AdGroupResource(it.id, it.campaignId, multiplier)
        }
    }

    override fun fillExportObject(resource: Multipliers, builder: ru.yandex.adv.direct.adgroup.AdGroup.Builder) {
        builder.addAllMultipliers(resource)
    }
}
