package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.SerpPlacementType
import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPlacementTypes
import ru.yandex.direct.core.entity.campaign.model.PlacementType
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupResourceType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupObject
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.AdGroupResource
import ru.yandex.direct.multitype.entity.LimitOffset

@Component
class AdGroupSerpPlacementTypeHandler(
    private val adGroupRepository: AdGroupRepository,
    private val campaignTypedRepository: CampaignTypedRepository
) : AdGroupBaseHandler<SerpPlacementType>() {
    override fun getAdGroupIdsToLoad(shard: Int, objects: Collection<BsExportAdGroupObject>): List<Long> {
        val adGroupIds = objects.mapNotNull { it.adGroupId }
        val campaignIds = objects.filter { it.adGroupId == null }.map { it.campaignId!! }

        return if (campaignIds.isNotEmpty()) {
            val selectionCriteria = AdGroupsSelectionCriteria().withCampaignIds(campaignIds.toSet())
            adGroupIds + adGroupRepository.getAdGroupIdsBySelectionCriteria(shard, selectionCriteria, LimitOffset.maxLimited())
        } else adGroupIds
    }

    override fun resourceType() = AdGroupResourceType.SERP_PLACEMENT_TYPE

    override fun loadResources(shard: Int, adGroups: Collection<AdGroup>): List<AdGroupResource<SerpPlacementType>> {
        val campaigns = campaignTypedRepository
            .getSafely(shard, adGroups.map { it.campaignId }, CampaignWithPlacementTypes::class.java)
            .associateBy { it.id }

        return adGroups
            .mapNotNull {
                val campaign = campaigns[it.campaignId] ?: return@mapNotNull null
                AdGroupResource(
                    adGroupId = it.id,
                    campaignId = it.campaignId,
                    resource = getSerpPlacementType(campaign.placementTypes)
                )
            }
    }

    override fun fillExportObject(resource: SerpPlacementType, builder: ru.yandex.adv.direct.adgroup.AdGroup.Builder) {
        builder.serpPlacementType = resource.number
    }

    private fun getSerpPlacementType(placementTypes: Set<PlacementType>): SerpPlacementType {
        return when (placementTypes.singleOrNull()) {
            null -> SerpPlacementType.SERP_PLACEMENT_TYPE_UNSPECIFIED
            PlacementType.SEARCH_PAGE -> SerpPlacementType.SERP_PLACEMENT_TYPE_NO_PRODUCT_GALLERY
            PlacementType.ADV_GALLERY -> SerpPlacementType.SERP_PLACEMENT_TYPE_PRODUCT_GALLERY_ONLY
        }
    }
}
