package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.bsexport.repository.adgroup.showconditions.BsExportAdgroupShowConditionsGeoRepository
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsRetargetingRepository
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithPageBlocks
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingJoinType
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingMode
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.MobileInstalledAppsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.TimeAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.AdGroupAdditionalTargetingRepository
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPlacementTypes
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository
import ru.yandex.direct.dbschema.ppc.enums.AdgroupAdditionalTargetingsTargetingType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupResourceType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupObject
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.ShowConditionsInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.common.getTargetTags
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.AdGroupResource
import ru.yandex.direct.logicprocessor.processors.bsexport.common.ShowConditionUtils.calcContextID
import ru.yandex.direct.logicprocessor.processors.bsexport.common.ShowConditionUtils.targetingExpressionSort
import ru.yandex.direct.logicprocessor.processors.bsexport.common.ShowConditionUtils.toAtom
import ru.yandex.direct.logicprocessor.processors.bsexport.common.TimeTargetBsExportUtils.addTargetTimeCondition
import ru.yandex.direct.regions.GeoTreeFactory
import ru.yandex.direct.utils.DateTimeUtils.moscowDateTimeToEpochSecond
import ru.yandex.grut.objects.proto.PlaceIds.EPlaceId.PI_DIRECT
import ru.yandex.grut.objects.proto.PlaceIds.EPlaceId.PI_REACH
import java.time.Duration

data class TargetingKey(val adGroupId: Long, val cls: Class<out AdGroupAdditionalTargeting>)

@Component
@Suppress("UNUSED_PARAMETER")
class AdGroupShowConditionsHandler(
    private val adGroupRepository: AdGroupRepository,
    private val mobileContentRepository: MobileContentRepository,
    private val campaignRepository: CampaignRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val bidsRetargetingRepository: BsExportBidsRetargetingRepository,
    private val showConditionsGeoRepository: BsExportAdgroupShowConditionsGeoRepository,
    private val targetingRepository: AdGroupAdditionalTargetingRepository,
    private val geoTreeFactory: GeoTreeFactory,
    private val additionalTargetingTypeConfigGenerator: AdditionalTargetingTypeConfigGenerator,
    ppcPropertiesSupport: PpcPropertiesSupport
) : AdGroupBaseHandler<TargetingExpression>() {

    private val oldRfExportByFilterEnabledProp = ppcPropertiesSupport.get(
        PpcPropertyNames.BS_EXPORT_OLD_RF_FILTER_INTERNAL_AD_GROUPS_ENABLED,
        Duration.ofMinutes(5),
    )
    private val filterByCampaignIdsProp = ppcPropertiesSupport.get(
        PpcPropertyNames.BS_EXPORT_OLD_RF_FILTER_BY_CAMPAIGN_IDS,
        Duration.ofMinutes(5),
    )
    private val ignoreMinusGeoCidPercentProp = ppcPropertiesSupport.get(
        PpcPropertyNames.IGNORE_MINUS_GEO_IN_ADGROUP_CONDITIONS_CID_PERCENT,
        Duration.ofMinutes(5),
    )
    private val ignoreMinusGeoCidsProp = ppcPropertiesSupport.get(
        PpcPropertyNames.IGNORE_MINUS_GEO_IN_ADGROUP_CONDITIONS_CIDS,
        Duration.ofMinutes(5),
    )

    companion object {
        private val directPlaceId: Long = PI_DIRECT.number.toLong()
        private val reachDirectPlaceId: Long = PI_REACH.number.toLong()
        private const val MOON_REGION_ID: Long = 20001

        fun needExportGroupRfCondition(
            oldRfExportByFilterEnabledProp: PpcProperty<Boolean>,
            filterByCampaignIdsProp: PpcProperty<Set<Long>>,
            campaignId: Long?
        ): Boolean {
            if (oldRfExportByFilterEnabledProp.getOrDefault(false)) {
                return filterByCampaignIdsProp.getOrDefault(emptySet()).contains(campaignId)
            }
            return true
        }
    }

    override fun resourceType() = AdGroupResourceType.SHOW_CONDITIONS

    override fun getAdGroupIdsToLoad(shard: Int, objects: Collection<BsExportAdGroupObject>): List<Long> {
        val groupIds = objects.mapNotNullTo(hashSetOf()) { it.adGroupId }

        val campaignIds = objects.mapNotNull { it.campaignId }
        groupIds += adGroupRepository.getAdGroupIdsByCampaignIds(shard, campaignIds).values.flatten().toSet()

        val bannerIds = objects.mapNotNull { (it.additionalInfo as? ShowConditionsInfo)?.bannerId }
        groupIds += adGroupRepository.getAdGroupIdsByBannerIds(shard, bannerIds)

        val mobileContentIds = objects.mapNotNull { (it.additionalInfo as? ShowConditionsInfo)?.mobileContentId }
        groupIds += mobileContentRepository.getAdgroupIdsForMobileContentIds(shard, mobileContentIds).toSet()

        val mobileContents = mobileContentRepository.getMobileContent(shard, mobileContentIds)
        groupIds += targetingRepository
            .getByClientIdsAndType(
                shard, mobileContents.map { it.clientId },
                AdgroupAdditionalTargetingsTargetingType.mobile_installed_apps
            ).filterIsInstance<MobileInstalledAppsAdGroupAdditionalTargeting>()
            .filter { it.value.any { app -> app.mobileContentId in mobileContentIds } }
            .map { it.adGroupId }

        return groupIds.toList()
    }

    private val conditionHandlers = listOf(
        ::addAdditionalTargetingConditions,
        ::addGeoConditions,
        ::addGoalContextConditions,
        ::addInternalConditions,
        ::addMobileContentConditions,
        ::addPageBlocksConditions,
        ::addPerformanceConditions,
        ::addPlaceConditions,
        ::addTargetTagsConditions,
        ::addTargetTimeCondition
    )

    override fun loadResources(shard: Int, adGroups: Collection<AdGroup>): List<AdGroupResource<TargetingExpression>> {
        val conditionBuilders = adGroups.associate { it.id to TargetingExpression.newBuilder() }

        conditionHandlers.forEach {
            it(shard, adGroups, conditionBuilders)
        }

        return adGroups.map { adGroup ->
            AdGroupResource(adGroup.id, adGroup.campaignId, conditionBuilders[adGroup.id]!!.build())
        }
    }

    override fun fillExportObject(
        resource: TargetingExpression,
        builder: ru.yandex.adv.direct.adgroup.AdGroup.Builder
    ) {
        builder.showConditions = targetingExpressionSort(resource)
        builder.contextId = calcContextID(builder.showConditions)
    }

    private fun addTargetTimeCondition(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        val timeTargetsWithItsGroups = targetingRepository
            .getByAdGroupIdsAndType(shard, adGroups.map { group -> group.id }, AdgroupAdditionalTargetingsTargetingType.timetarget)
            .map { l -> l as TimeAdGroupAdditionalTargeting }
            .map { l -> l.adGroupId to l.value }

        for ((adGroupId, timeTargets) in timeTargetsWithItsGroups) {
            timeTargets.forEach { tt -> builders[adGroupId]?.let { addTargetTimeCondition(tt, it) } }
        }
    }

    private fun addAdditionalTargetingConditions(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        val targetingsMap = targetingRepository.getByAdGroupIds(shard, adGroups.map { it.id })
            .groupBy { TargetingKey(it.adGroupId, it.javaClass) }
        val targetingsConfig = additionalTargetingTypeConfigGenerator.getTargetingsConfig(shard, targetingsMap)
        val groupTypes = adGroups.flatMap { gr -> targetingsConfig.keys.map { type -> Pair(gr.id, type) } }
        for ((adGroupId, type) in groupTypes) {
            val builder = builders[adGroupId]!!
            val config = targetingsConfig[type]
            if (config != null) {
                targetingsMap[TargetingKey(adGroupId, config.cls)]
                    ?.forEach { targeting ->
                        val operation = if (targeting.targetingMode == AdGroupAdditionalTargetingMode.TARGETING)
                            config.targetingOperation else config.filteringOperation
                        val andBuilder = TargetingExpression.Disjunction.newBuilder()

                        for ((value, keyword) in config.getValuesKeywordsWithCast(targeting)) {
                            val atom = toAtom(keyword, operation, value)
                            if (targeting.joinType == AdGroupAdditionalTargetingJoinType.ALL) {
                                builder.addAnd(TargetingExpression.Disjunction.newBuilder().addOr(atom))
                            } else {
                                andBuilder.addOr(atom)
                            }
                        }

                        if (andBuilder.orCount > 0) {
                            builder.addAnd(andBuilder)
                        }
                    }
            }
        }
    }

    private fun addGeoConditions(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        showConditionsGeoRepository
            .getGeoInfoByAdGroupIds(shard, adGroups.map { it.id })
            .forEach { (adGroupId, geoInfo) ->
                val ignoreMinusGeo =
                    (geoInfo.campaignId % 100 < ignoreMinusGeoCidPercentProp.getOrDefault(0)) ||
                        geoInfo.campaignId in ignoreMinusGeoCidsProp.getOrDefault(emptySet())
                val newGeo = if (ignoreMinusGeo) {
                    geoInfo.geo
                } else {
                    geoTreeFactory.apiGeoTree.excludeRegions(geoInfo.geo, geoInfo.minusGeo)
                }
                if (newGeo.size == 1 && newGeo[0] == 0L) {
                    return@forEach
                }
                var (includeGeo, excludeGeo) = newGeo.filter { it != 0L }.partition { it > 0 }
                if (newGeo.isEmpty()) {
                    includeGeo = listOf(MOON_REGION_ID)
                }

                if (includeGeo.isNotEmpty()) {
                    builders[adGroupId]!!.addAnd(
                        TargetingExpression.Disjunction.newBuilder().addAllOr(
                            includeGeo.sorted().map {
                                toAtom(KeywordEnum.RegId, OperationEnum.Equal, it)
                            }
                        )
                    )
                }
                excludeGeo.sorted().forEach {
                    builders[adGroupId]!!.addAnd(
                        TargetingExpression.Disjunction.newBuilder().addOr(
                            toAtom(KeywordEnum.RegId, OperationEnum.NotEqual, -it)
                        )
                    )
                }
            }
    }

    private fun addGoalContextConditions(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        val internalAdGroupIds = adGroups.filter { it.type == AdGroupType.INTERNAL }.map { it.id }
        bidsRetargetingRepository
            .getBidsByPids(shard, internalAdGroupIds)
            .associateBy { it.adGroupId }
            .forEach { (adGroupId, bid) ->
                builders[adGroupId]!!.addAnd(
                    TargetingExpression.Disjunction.newBuilder().addOr(
                        toAtom(KeywordEnum.GoalContextId, OperationEnum.MatchGoalContext, bid.retCondId)
                    )
                )
            }
    }

    private fun addInternalConditions(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        adGroups
            .filterIsInstance<InternalAdGroup>()
            .forEach { adGroup ->
                val builder = builders[adGroup.id]!!
                if (adGroup.startTime != null) {
                    builder.addAnd(
                        TargetingExpression.Disjunction.newBuilder().addOr(
                            toAtom(
                                KeywordEnum.Unixtime, OperationEnum.GreaterOrEqual,
                                moscowDateTimeToEpochSecond(adGroup.startTime)
                            )
                        )
                    )
                }
                if (adGroup.finishTime != null) {
                    builder.addAnd(
                        TargetingExpression.Disjunction.newBuilder().addOr(
                            toAtom(
                                KeywordEnum.Unixtime, OperationEnum.Less,
                                moscowDateTimeToEpochSecond(adGroup.finishTime)
                            )
                        )
                    )
                }
                if ((adGroup.rf ?: 0) > 0 && needExportGroupRfCondition(oldRfExportByFilterEnabledProp,
                        filterByCampaignIdsProp, adGroup.campaignId)) {
                    builder.addAnd(
                        TargetingExpression.Disjunction.newBuilder().addOr(
                            toAtom(KeywordEnum.FrequencyDay, OperationEnum.Less, adGroup.rf)
                        ).addOr(
                            toAtom(KeywordEnum.FreqExpire, OperationEnum.GreaterOrEqual, (adGroup.rfReset ?: 0))
                        )
                    )
                }
            }
    }

    private fun addMobileContentConditions(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        adGroups
            .filterIsInstance<MobileContentAdGroup>()
            .forEach { adGroup ->
                val builder = builders[adGroup.id]!!
                MobileAppsCommon.addOsTypeCondition(adGroup, builder)
                MobileAppsCommon.addContentStoreCondition(adGroup, builder)
                MobileAppsCommon.addMobileOsMinVersionCondition(adGroup, builder)
                MobileAppsCommon.addCellCondition(adGroup, builder)
                MobileAppsCommon.addDeviceTypeCondition(adGroup, builder)
            }
    }

    private fun addPageBlocksConditions(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        adGroups
            .filterIsInstance<AdGroupWithPageBlocks>()
            .filter { !it.pageBlocks.isNullOrEmpty() }
            .forEach { adGroup ->
                builders[adGroup.id]!!.addAnd(
                    TargetingExpression.Disjunction.newBuilder().addAllOr(
                        adGroup.pageBlocks.map {
                            toAtom(
                                KeywordEnum.PageIdImpId,
                                OperationEnum.Match,
                                "${it.pageId}:${it.impId}"
                            )
                        }
                    )
                )
            }
    }

    private fun addPerformanceConditions(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        adGroups
            .filter { it.type == AdGroupType.PERFORMANCE }
            .forEach { adGroup ->
                builders[adGroup.id]!!.addAnd(
                    TargetingExpression.Disjunction.newBuilder().addOr(
                        toAtom(KeywordEnum.Performance, OperationEnum.Equal, "1")
                    )
                )
            }
    }

    private fun addPlaceConditions(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        val campaignIds = adGroups.map { it.campaignId }.distinct()

        val campaignTypesIdMap = campaignRepository.getCampaignsTypeMap(shard, campaignIds)
        val internalAdGroupIds = adGroups
            .filter {
                it.campaignId in campaignTypesIdMap
                    && campaignTypesIdMap[it.campaignId] in CampaignTypeKinds.INTERNAL
            }.map { it.id }
        val campaignInternalPlaces = campaignRepository.getCampaignInternalPlaces(shard, campaignIds)

        adGroups
            .filter { it.campaignId in campaignTypesIdMap }
            .forEach { adGroup ->
                builders[adGroup.id]!!.addAnd(
                    TargetingExpression.Disjunction.newBuilder().addOr(
                        toAtom(
                            KeywordEnum.PlaceId,
                            OperationEnum.Equal,
                            if (adGroup.id in internalAdGroupIds) {
                                campaignInternalPlaces[adGroup.campaignId]!!
                            } else if (campaignTypesIdMap[adGroup.campaignId] in CampaignTypeKinds.CPM) {
                                reachDirectPlaceId
                            } else {
                                directPlaceId
                            }
                        )
                    )
                )
            }
    }

    private fun addTargetTagsConditions(
        shard: Int,
        adGroups: Collection<AdGroup>,
        builders: Map<Long, TargetingExpression.Builder>
    ) {
        val campaigns = campaignTypedRepository
            .getSafely(shard, adGroups.map { it.campaignId }, CampaignWithPlacementTypes::class.java)
            .associateBy { it.id }

        adGroups.forEach { adGroup ->
            val tags = getTargetTags(adGroup, campaigns[adGroup.campaignId])
            if (tags.isNullOrEmpty())
                return@forEach

            builders[adGroup.id]!!.addAnd(
                TargetingExpression.Disjunction.newBuilder().addAllOr(
                    tags.map { toAtom(KeywordEnum.TargetTags, OperationEnum.Match, it) }
                )
            )
        }
    }
}
