package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler

import com.google.common.base.Suppliers
import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.AdGroupShowConditionType
import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingMode
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AuditoriumGeoSegmentsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.CallerReferrersAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.ClidTypesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.ClidsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.ContentCategoriesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.DesktopInstalledAppsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.DeviceIdsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.FeaturesInPPAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.HasLCookieAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.HasPassportIdAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.InterfaceLangsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.InternalNetworkAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsDefaultYandexSearchAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsPPLoggedInAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsVirusedAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsYandexPlusAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.MobileInstalledAppsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.PlusUserSegmentsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.QueryOptionsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.QueryReferersAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.SearchTextAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.ShowDatesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.SidsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.TestIdsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.UserAgentsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.UuidsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.VisitGoalsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.YandexUidsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.YandexuidAgeAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.YpCookiesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.YsCookiesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.BrowserEnginesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.BrowserNamesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.DeviceNamesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.DeviceVendorsAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.IsMobileAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.IsTabletAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.IsTouchAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.OsFamiliesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.OsNamesAdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.UatraitsTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.VersionedTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.VersionedTargetingHelper
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.dbschema.ppc.enums.AdgroupAdditionalTargetingsTargetingType
import java.util.concurrent.TimeUnit

/**
 * Генератор настроек, по которым будут собираться условия показа типа Additional Targeting
 */
@Component
class AdditionalTargetingTypeConfigGenerator(
    private val cryptaSegmentRepository: CryptaSegmentRepository,
    private val mobileContentRepository: MobileContentRepository,
) {

    fun mobileInstalledAppsFixedKeywordList(
        mobileContents: Map<Long, MobileContent>,
    ): (MobileInstalledAppsAdGroupAdditionalTargeting) -> Collection<Pair<String, KeywordEnum>> {

        val getListValues: (MobileInstalledAppsAdGroupAdditionalTargeting) -> Collection<String> = {
            it.value.map { app -> MobileAppsCommon.mobileAppsTargetingToStr(mobileContents[app.mobileContentId]!!) }
        }

        return { targeting ->
            val result = getListValues(targeting).map { it to KeywordEnum.ExceptAppsOnCpi }
            // При антитаргетинге вместе с 395ым должен оправляться 723й для более точного результата
            if (targeting.targetingMode == AdGroupAdditionalTargetingMode.FILTERING) {
                result.plus(getListValues(targeting).map { it to KeywordEnum.CryptaAdditionalApps })
            } else {
                result
            }
        }
    }

    private val segmentMap = Suppliers.memoizeWithExpiration(
        cryptaSegmentRepository::getContentSegments,
        15, TimeUnit.MINUTES
    )

    private fun <T : AdGroupAdditionalTargeting> fixedKeywordList(
        keyword: KeywordEnum,
        getListValues: (T) -> Collection<String>
    ): (T) -> Collection<Pair<String, KeywordEnum>> {
        return { targeting -> getListValues(targeting).map { it to keyword } }
    }

    private fun <T : AdGroupAdditionalTargeting> fixedValueKeyword(
        keyword: KeywordEnum,
        getValue: (T) -> String
    ): (T) -> Collection<Pair<String, KeywordEnum>> {
        return { targeting -> listOf(getValue(targeting) to keyword) }
    }

    private fun booleanKeyword(keyword: KeywordEnum, targetingValue: String, filteringValue: String):
        (AdGroupAdditionalTargeting) -> Collection<Pair<String, KeywordEnum>> {
        return { t ->
            if (t.targetingMode == AdGroupAdditionalTargetingMode.TARGETING) {
                listOf(targetingValue to keyword)
            } else {
                listOf(filteringValue to keyword)
            }
        }
    }

    private fun versionedTargetingToStr(t: VersionedTargeting) =
        "%s:%s:%s".format(
            t.targetingValueEntryId?.toString() ?: "",
            if (t.minVersion == null) "" else VersionedTargetingHelper.versionToInt(t.minVersion).toString(),
            if (t.maxVersion == null) "" else VersionedTargetingHelper.versionToInt(t.maxVersion).toString(),
        )

    private fun uatraitsTargetingToStr(t: UatraitsTargeting) = t.targetingValueEntryId.toString()

    fun getTargetingsConfig(shard: Int, showConditionInfo: Map<TargetingKey, List<AdGroupAdditionalTargeting>>)
        : Map<AdgroupAdditionalTargetingsTargetingType, TargetingTypeConfig<out AdGroupAdditionalTargeting>> {
        val mobileContentIds = showConditionInfo.values.flatten()
            .filterIsInstance(MobileInstalledAppsAdGroupAdditionalTargeting::class.java)
            .flatMap { it.value }
            .map { it.mobileContentId }
            .toSet()
        val mobileContents = mobileContentRepository.getMobileContent(shard, mobileContentIds).associateBy { it.id }

        val cryptaTypeToKeyword = mapOf(
            GoalType.CONTENT_GENRE to KeywordEnum.ContentGenre,
            GoalType.CONTENT_CATEGORY to KeywordEnum.ContentCategory,
        )

        // Copied from https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/BS/Export.pm?rev=r8433707#L290
        return mapOf(
            AdgroupAdditionalTargetingsTargetingType.yandexuids to TargetingTypeConfig(
                YandexUidsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_YANDEXUIDS,
                OperationEnum.Like,
                OperationEnum.NotLike,
                fixedKeywordList(KeywordEnum.UniqId) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.device_names to TargetingTypeConfig(
                DeviceNamesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_DEVICE_NAMES,
                OperationEnum.IcaseMatch,
                OperationEnum.IcaseNotMatch,
                fixedKeywordList(KeywordEnum.UatraitsDeviceName) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.interface_langs to TargetingTypeConfig(
                InterfaceLangsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_INTERFACE_LANGS,
                OperationEnum.IcaseMatch,
                OperationEnum.IcaseNotMatch,
                fixedKeywordList(KeywordEnum.Lang) {
                    it.value.map { lang -> lang.typedValue }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.desktop_installed_apps to TargetingTypeConfig(
                DesktopInstalledAppsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_DESKTOP_INSTALLED_APPS,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
                fixedKeywordList(KeywordEnum.InstalledYasoft) {
                    it.value.map { l -> l.toString() }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.query_referers to TargetingTypeConfig(
                QueryReferersAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_QUERY_REFERERS,
                OperationEnum.Like,
                OperationEnum.NotLike,
                fixedKeywordList(KeywordEnum.Referer) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.user_agent to TargetingTypeConfig(
                UserAgentsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_USER_AGENT,
                OperationEnum.Like,
                OperationEnum.NotLike,
                fixedKeywordList(KeywordEnum.UserAgent) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.show_dates to TargetingTypeConfig(
                ShowDatesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_SHOW_DATES,
                OperationEnum.Like,
                OperationEnum.NotLike,
                fixedKeywordList(KeywordEnum.Timestamp) {
                    it.value.map { l -> l.toString() }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.query_options to TargetingTypeConfig(
                QueryOptionsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_QUERY_OPTIONS,
                OperationEnum.IcaseMatch,
                OperationEnum.IcaseNotMatch,
                fixedKeywordList(KeywordEnum.Options) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.clids to TargetingTypeConfig(
                ClidsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_CLIDS,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
                fixedKeywordList(KeywordEnum.StatId) {
                    it.value.map { l -> l.toString() }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.clid_types to TargetingTypeConfig(
                ClidTypesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_CLID_TYPES,
                OperationEnum.Match,
                OperationEnum.NotMatch,
                fixedKeywordList(KeywordEnum.ClidType) {
                    it.value.map { l -> l.toString() }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.test_ids to TargetingTypeConfig(
                TestIdsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_TEST_IDS,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
                fixedKeywordList(KeywordEnum.TestIds) {
                    it.value.map { l -> l.toString() }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.ys_cookies to TargetingTypeConfig(
                YsCookiesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_YS_COOKIES,
                OperationEnum.Like,
                OperationEnum.NotLike,
                fixedKeywordList(KeywordEnum.CookieYs) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.yp_cookies to TargetingTypeConfig(
                YpCookiesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_YP_COOKIES,
                OperationEnum.Like,
                OperationEnum.NotLike,
                fixedKeywordList(KeywordEnum.CookieYp) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.internal_network to TargetingTypeConfig(
                InternalNetworkAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_INTERNAL_NETWORK,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
                booleanKeyword(
                    keyword = KeywordEnum.NetworkId,
                    targetingValue = "2",
                    filteringValue = "2",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.is_mobile to TargetingTypeConfig(
                IsMobileAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_IS_MOBILE,
                OperationEnum.Equal,
                OperationEnum.Equal,
                booleanKeyword(
                    keyword = KeywordEnum.DeviceIsMobile,
                    targetingValue = "1",
                    filteringValue = "0",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.has_l_cookie to TargetingTypeConfig(
                HasLCookieAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_HAS_L_COOKIE,
                OperationEnum.NotEqual,
                OperationEnum.Equal,
                booleanKeyword(
                    keyword = KeywordEnum.CookieL,
                    targetingValue = "0",
                    filteringValue = "0",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.has_passport_id to TargetingTypeConfig(
                HasPassportIdAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_HAS_PASSPORT_ID,
                OperationEnum.Greater,
                OperationEnum.Equal,
                booleanKeyword(
                    keyword = KeywordEnum.PassportId,
                    targetingValue = "0",
                    filteringValue = "0",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.is_pp_logged_in to TargetingTypeConfig(
                IsPPLoggedInAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_IS_PP_LOGGED_IN,
                OperationEnum.Equal,
                OperationEnum.Equal,
                booleanKeyword(
                    keyword = KeywordEnum.Authorized,
                    targetingValue = "1",
                    filteringValue = "0",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.is_tablet to TargetingTypeConfig(
                IsTabletAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_IS_TABLET,
                OperationEnum.Equal,
                OperationEnum.Equal,
                booleanKeyword(
                    keyword = KeywordEnum.DeviceIsTablet,
                    targetingValue = "1",
                    filteringValue = "0",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.is_touch to TargetingTypeConfig(
                IsTouchAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_IS_TOUCH,
                OperationEnum.Equal,
                OperationEnum.Equal,
                booleanKeyword(
                    keyword = KeywordEnum.DeviceIsTouch,
                    targetingValue = "1",
                    filteringValue = "0",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.is_virused to TargetingTypeConfig(
                IsVirusedAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_IS_VIRUSED,
                OperationEnum.Equal,
                OperationEnum.Equal,
                booleanKeyword(
                    keyword = KeywordEnum.VirusMark,
                    targetingValue = "1",
                    filteringValue = "0",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.is_yandex_plus to TargetingTypeConfig(
                IsYandexPlusAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_IS_YANDEX_PLUS,
                OperationEnum.Equal,
                OperationEnum.Equal,
                booleanKeyword(
                    keyword = KeywordEnum.YandexPlusEnabled,
                    targetingValue = "1",
                    filteringValue = "0",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.is_default_yandex_search to TargetingTypeConfig(
                IsDefaultYandexSearchAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_IS_DEFAULT_YANDEX_SEARCH,
                OperationEnum.Equal,
                OperationEnum.Equal,
                booleanKeyword(
                    keyword = KeywordEnum.SearchAntitargeting,
                    targetingValue = "1",
                    filteringValue = "0",
                )
            ),
            AdgroupAdditionalTargetingsTargetingType.visit_goals to TargetingTypeConfig(
                VisitGoalsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_VISIT_GOALS,
                OperationEnum.Match,
                OperationEnum.NotMatch,
                fixedKeywordList(KeywordEnum.VisitGoal) {
                    it.value.map { l -> l.toString() }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.auditorium_geosegments to TargetingTypeConfig(
                AuditoriumGeoSegmentsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_AUDITORIUM_GEOSEGMENTS,
                OperationEnum.Match,
                OperationEnum.NotMatch,
                fixedKeywordList(KeywordEnum.AuditoriumGeosegments) {
                    it.value.map { l -> l.toString() }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.sids to TargetingTypeConfig(
                SidsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_SIDS,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
                fixedKeywordList(KeywordEnum.Sids) {
                    it.value.map { l -> l.toString() }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.device_id to TargetingTypeConfig(
                DeviceIdsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_DEVICE_ID,
                OperationEnum.IcaseMatch,
                OperationEnum.IcaseNotMatch,
                fixedKeywordList(KeywordEnum.DeviceId) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.uuid to TargetingTypeConfig(
                UuidsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_UUID,
                OperationEnum.IcaseMatch,
                OperationEnum.IcaseNotMatch,
                fixedKeywordList(KeywordEnum.Uuid) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.plus_user_segments to TargetingTypeConfig(
                PlusUserSegmentsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_PLUS_USER_SEGMENTS,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
                fixedKeywordList(KeywordEnum.PlusUserSegments) {
                    it.value.map { l -> l.toString() }
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.search_text to TargetingTypeConfig(
                SearchTextAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_SEARCH_TEXT,
                OperationEnum.IcaseMatch,
                OperationEnum.IcaseNotMatch,
                fixedKeywordList(KeywordEnum.SearchText) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.features_in_pp to TargetingTypeConfig(
                FeaturesInPPAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_FEATURES_IN_PP,
                OperationEnum.IcaseMatch,
                OperationEnum.IcaseNotMatch,
                fixedKeywordList(KeywordEnum.EnabledFeatures) {
                    it.value
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.browser_names to TargetingTypeConfig(
                BrowserNamesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_BROWSER_NAMES,
                OperationEnum.MatchNameAndVersionRange,
                OperationEnum.NotMatchNameAndVersionRange,
                fixedKeywordList(KeywordEnum.BrowserNameAndVersion) {
                    it.value.map(::versionedTargetingToStr)
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.browser_engines to TargetingTypeConfig(
                BrowserEnginesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_BROWSER_ENGINES,
                OperationEnum.MatchNameAndVersionRange,
                OperationEnum.NotMatchNameAndVersionRange,
                fixedKeywordList(KeywordEnum.BrowserEngineNameAndVersion) {
                    it.value.map(::versionedTargetingToStr)
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.os_families to TargetingTypeConfig(
                OsFamiliesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_OS_FAMILIES,
                OperationEnum.MatchNameAndVersionRange,
                OperationEnum.NotMatchNameAndVersionRange,
                fixedKeywordList(KeywordEnum.OsFamilyAndVersion) {
                    it.value.map(::versionedTargetingToStr)
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.os_names to TargetingTypeConfig(
                OsNamesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_OS_NAMES,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
                fixedKeywordList(KeywordEnum.OsName) {
                    it.value.map(::uatraitsTargetingToStr)
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.device_vendors to TargetingTypeConfig(
                DeviceVendorsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_DEVICE_VENDORS,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
                fixedKeywordList(KeywordEnum.Lang) {
                    it.value.map(::uatraitsTargetingToStr)
                },
            ),
            AdgroupAdditionalTargetingsTargetingType.mobile_installed_apps to TargetingTypeConfig(
                MobileInstalledAppsAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_MOBILE_INSTALLED_APPS,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
                mobileInstalledAppsFixedKeywordList(mobileContents),
            ),
            AdgroupAdditionalTargetingsTargetingType.yandexuid_age to TargetingTypeConfig(
                YandexuidAgeAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_YANDEXUID_AGE,
                OperationEnum.GreaterOrEqual,
                OperationEnum.LessOrEqual,
                fixedValueKeyword(KeywordEnum.YandexuidAge) {
                    // в БК отправляем в секундах, у нас храним в часах
                    it.value.map { hour -> (hour * 3600).toString() }
                }
            ),
            AdgroupAdditionalTargetingsTargetingType.content_categories to TargetingTypeConfig(
                ContentCategoriesAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_CONTENT_CATEGORIES,
                OperationEnum.Equal,
                OperationEnum.NotEqual,
            ) {
                it.value.map { goalId ->
                    val goal = segmentMap.get()[goalId]!!
                    Pair(goal.keywordValue, cryptaTypeToKeyword[goal.type]!!)
                }
            },
            AdgroupAdditionalTargetingsTargetingType.caller_referrers to TargetingTypeConfig(
                CallerReferrersAdGroupAdditionalTargeting::class.java,
                AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_CALLER_REFERRERS,
                OperationEnum.Like,
                OperationEnum.NotLike,
                fixedKeywordList(KeywordEnum.CallerReferer) {
                    it.value
                },
            ),
        )
    }
}

data class TargetingTypeConfig<T : AdGroupAdditionalTargeting>(
    val cls: Class<T>,
    val type: AdGroupShowConditionType,
    val targetingOperation: OperationEnum,
    val filteringOperation: OperationEnum,
    val getValuesKeywords: ((T) -> Collection<Pair<String, KeywordEnum>>),
) {
    @Suppress("UNCHECKED_CAST")
    fun getValuesKeywordsWithCast(t: AdGroupAdditionalTargeting): Collection<Pair<String, KeywordEnum>> {
        return getValuesKeywords(t as T)
    }
}
