package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler

import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupDeviceTypeTargeting
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupNetworkTargeting
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent
import ru.yandex.direct.core.entity.mobilecontent.model.OsType
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentServiceConstants
import ru.yandex.direct.core.validation.constraints.MobileContentConstraints
import ru.yandex.direct.logicprocessor.processors.bsexport.common.ShowConditionUtils.toAtom
import java.util.zip.CRC32

object MobileAppsCommon {
    private val versionRegex = "(?:\\.\\s*|\\s+)".toRegex()
    private const val maxMobileVersionParts = 2

    fun mobileAppsTargetingToStr(mobileContent: MobileContent): String {
        var appId = if (mobileContent.osType == OsType.ANDROID) mobileContent.storeContentId else mobileContent.bundleId
        if (appId == null || appId == "0") {
            appId = ""
        }
        val storeName = MobileContentServiceConstants.OS_TYPE_TO_STORE_NAME[mobileContent.osType]!!
        val crc = CRC32()
        crc.update((appId + storeName).toByteArray())
        return crc.value.toString()
    }

    fun getMobileOsMinVersion(adGroup: MobileContentAdGroup): Int {
        val targetingVersionParts = getVersionParts(adGroup.minimalOperatingSystemVersion)
        val contentVersionParts = getVersionParts(adGroup.mobileContent.minOsVersion)

        var maxParts = if (compareParts(targetingVersionParts, contentVersionParts) > 0) {
            targetingVersionParts
        } else {
            contentVersionParts
        }
        if (maxParts.all { it == 0 }) {
            val osVersions = MobileContentConstraints.OS_VERSIONS[adGroup.mobileContent.osType]!!
            val maxOsVersion = osVersions.maxWithOrNull { a, b -> compareParts(getVersionParts(a), getVersionParts(b)) }
            maxParts = getVersionParts(maxOsVersion!!)
        }
        return 1000 * maxParts[0] + minOf(999, maxParts[1])
    }

    fun getVersionParts(versionStrNullable: String?): List<Int> {
        val versionStr = versionStrNullable ?: ""
        val parts = versionStr.split(versionRegex, maxMobileVersionParts + 1).toMutableList()
        while (parts.size < maxMobileVersionParts) {
            parts.add("0")
        }
        return parts.slice(0 until maxMobileVersionParts).map {
            val x = it.toIntOrNull() ?: 0
            if (x >= 0) x else 0
        }
    }

    private fun compareParts(first: List<Int>, second: List<Int>): Int {
        for ((a, b) in first zip second) {
            if (a != b) {
                return a - b
            }
        }
        return 0
    }

    fun addOsTypeCondition(adGroup: MobileContentAdGroup, builder: TargetingExpression.Builder) {
        val osTypeToInt = mapOf(OsType.ANDROID to "2", OsType.IOS to "3")
        builder.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                toAtom(
                    KeywordEnum.DetailedDeviceType, OperationEnum.Equal,
                    osTypeToInt[adGroup.mobileContent.osType]!!
                )
            )
        )
    }

    fun addContentStoreCondition(adGroup: MobileContentAdGroup, builder: TargetingExpression.Builder) {
        builder.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                toAtom(
                    KeywordEnum.ExceptAppsOnCpi, OperationEnum.NotEqual,
                    mobileAppsTargetingToStr(adGroup.mobileContent)
                )
            )
        )
    }

    fun addMobileOsMinVersionCondition(adGroup: MobileContentAdGroup, builder: TargetingExpression.Builder) {
        builder.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                toAtom(KeywordEnum.OsMinVersion, OperationEnum.GreaterOrEqual, getMobileOsMinVersion(adGroup))
            )
        )
    }

    fun addCellCondition(adGroup: MobileContentAdGroup, builder: TargetingExpression.Builder) {
        if (adGroup.networkTargeting.contains(MobileContentAdGroupNetworkTargeting.CELLULAR))
            return
        builder.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                toAtom(KeywordEnum.StationaryConnection, OperationEnum.Match, "1")
            )
        )
    }

    fun addDeviceTypeCondition(adGroup: MobileContentAdGroup, builder: TargetingExpression.Builder) {
        if (adGroup.deviceTypeTargeting.isNullOrEmpty())
            return
        val targetingToInt = mapOf(
            MobileContentAdGroupDeviceTypeTargeting.PHONE to "3",
            MobileContentAdGroupDeviceTypeTargeting.TABLET to "4",
        )
        builder.addAnd(
            TargetingExpression.Disjunction.newBuilder().addAllOr(
                adGroup.deviceTypeTargeting.map {
                    toAtom(KeywordEnum.DeviceType, OperationEnum.Equal, targetingToInt[it]!!)
                }
            )
        )
    }
}
