package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler.minusphrases

import java.util.concurrent.Callable
import java.util.concurrent.ForkJoinPool
import java.util.regex.Pattern
import java.util.stream.Collectors
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.advq.query.QueryParser
import ru.yandex.advq.query.QuerySyntax
import ru.yandex.advq.query.ast.Expression
import ru.yandex.advq.query.ast.ExpressionStopWordsTransform
import ru.yandex.advq.query.ast.ExpressionTransform
import ru.yandex.advq.query.ast.Word
import ru.yandex.advq.query.ast.WordKind
import ru.yandex.direct.core.bsexport.repository.adgroup.resources.BsExportAdgroupMinusPhrasesRepository
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer
import ru.yandex.direct.core.entity.stopword.service.StopWordService
import ru.yandex.direct.libs.keywordutils.parser.KeywordBuilder
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler.AdGroupMinusPhrasesHandler
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler.minusphrases.MinusPhrasesBsPrepareService.Companion.SPECIAL_SYMBOLS
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.utils.TextConstants

typealias MinusPhrasesList = List<String>

data class AdGroupMinusPhrasesInfo(
    val adGroupId: Long,
    val campaignId: Long,
    val minusPhrases: MinusPhrasesList,
)

@Component
class MinusPhrasesBsPrepareService(
    private val wordsNormalizer: KeywordNormalizer,
    private val stopWordService: StopWordService,
    private val minusPhrasesRepository: BsExportAdgroupMinusPhrasesRepository,
) : AutoCloseable {
    companion object {
        const val SPECIAL_SYMBOLS = "+!"
        private const val ALLOWED_NOT_SPACIAL_SYMBOLS = "[].\"-"
        private const val ALLOWED_SYMBOLS = "$ALLOWED_NOT_SPACIAL_SYMBOLS $SPECIAL_SYMBOLS"
        private val NOT_ALLOW_KEYWORD_LETTERS_REGEX =
            "[^${Pattern.quote(TextConstants.LETTERS_AND_NUMBERS + ALLOWED_SYMBOLS)}]".toRegex()

        private val ONE_OR_MORE_SPACES_REGEX = "\\s+".toRegex()
        private val SPECIAL_SYMBOLS_WITH_NEXT_DISALLOWED_SYMBOLS_OR_EOL_REGEX =
            ("[$SPECIAL_SYMBOLS]" +
                "([^${Pattern.quote(TextConstants.LETTERS_AND_NUMBERS + ALLOWED_NOT_SPACIAL_SYMBOLS)}]+|$)")
                .toRegex()
        private val logger = LoggerFactory.getLogger(MinusPhrasesBsPrepareService::class.java)
    }

    private val minusPhrasesProcessJoinPool =
        ForkJoinPool(10, getForkJoinPoolFactory(), null, false)

    private val queryParser = QueryParser.newBuilder(QuerySyntax.DIRECT)
        .setSplitCompoundWords(false)
        .setOptimize(false)
        .setPrepareQueries(false)
        .build()

    /**
     * Преобразует минус фразы:
     * Для каждой минус фразы:
     *      * Удаляет символ - в начале минус фразы, а так же в начале и в конце слов
     *      * Удаляет все недопустимые символы
     *      * множественные пробелы заменяет на один
     *      * удаляет краевые пробелы
     *      * удаляет краевые точки
     *      * заменяет точки на пробел, если они не внутри чисел (например, 36.6)
     *      * многоточия заменят на точку
     *      * перед стоп словами ставит "!", перед стоп словами во фразах в кавычках ставит +, в квадратных скобках не проверяет стоп слова
     *      * кавычки заменяет на добавление ~0 в конце строки
     *
     * Из полученных фраз выбирает уникальные по нормальной форме и сортирует
     *
     * Оригинал https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/MinusWords.pm?rev=r5602766#L183
     */

    fun processAdGroupsMinusPhrases(shard: Int, adGroups: Collection<AdGroup>): List<AdGroupMinusPhrasesInfo> {

        val minusKeywordIds = adGroups
            .flatMap {
                it.libraryMinusKeywordsIds + it.minusKeywordsId
            }
            .distinct()
        val minusPhrasesMap = minusPhrasesRepository.getMinusPhrases(shard, minusKeywordIds)
        return processAdGroupsMinusPhrases(adGroups, minusPhrasesMap)

    }

    fun processAdGroupsMinusPhrases(adGroups: Collection<AdGroup>, minusPhrasesMap: Map<Long, MinusPhrasesList>): List<AdGroupMinusPhrasesInfo> {
        val (adGroupsWithOnlyPrivateMinusPhrases, adGroupsWithLibraryMinusPhrases) = adGroups
            .partition { it.libraryMinusKeywordsIds.isEmpty() }

        val adGroupsOnlyPrivateMinusPhrasesResources = getResourceForAdGroupsWithOnlyPrivateMinusPhrases(
            adGroupsWithOnlyPrivateMinusPhrases, minusPhrasesMap)
        val adGroupsWithLibraryMinusPhrasesResources = getResourceForAdGroupsWithLibraryMinusPhrases(
            adGroupsWithLibraryMinusPhrases, minusPhrasesMap)

        return adGroupsOnlyPrivateMinusPhrasesResources + adGroupsWithLibraryMinusPhrasesResources
    }

    fun processMinusPhrase(minusPhraseInput: String): PhraseToNormForm {
        var minusPhrase = minusPhraseInput
            .removePrefix("-").replace(" -", " ").replace("- ", " ")
            .replace(NOT_ALLOW_KEYWORD_LETTERS_REGEX, " ")
            // удаляет ! и + и все что за ними, если за ними не буквы, цифры, допустимые неслужебные символы
            .replace(SPECIAL_SYMBOLS_WITH_NEXT_DISALLOWED_SYMBOLS_OR_EOL_REGEX, " ")
            .replace(ONE_OR_MORE_SPACES_REGEX, " ")
            .removeSuffix(" ").removePrefix(" ")
            .removeSuffix(".").removePrefix(".")
            .replace("[ ", "[").replace(" ]", "]")
            .replace("\" ", "\"").replace(" \"", "\"")


        minusPhrase = Trace.current().profile("process_words_in_phrase").use {
            try {
                processWordsInPhrase(minusPhrase)
            } catch (e: Exception) {
                logger.error("Faild to process minus phrase words: $minusPhrase, exception: $e")
                ""
            }
        }
        if (minusPhrase.isEmpty()) return PhraseToNormForm("", "")

        val normMinusPhrase = wordsNormalizer.normalizeKeyword(minusPhrase)

        return PhraseToNormForm(minusPhrase, normMinusPhrase)
    }

    fun processWordsInPhrase(minusPhraseInput: String): String {
        if (minusPhraseInput.isEmpty()) return minusPhraseInput

        val expression = queryParser.parseQuery(minusPhraseInput)
            .accept(MarkStopWordsTransform(stopWordService))
            .accept(RemovePeriodTransform())
        val keyword = KeywordBuilder.from(expression).build()
        return keyword.toString()
    }

    private fun getResourceForAdGroupsWithOnlyPrivateMinusPhrases(
        adGroups: Collection<AdGroup>,
        minusPhrasesMap: Map<Long, MinusPhrasesList>): List<AdGroupMinusPhrasesInfo> {
        return adGroups
            .map {
                val adGroupMinusPhrases: List<String> =
                    if (it.minusKeywordsId != null && minusPhrasesMap.containsKey(it.minusKeywordsId))
                        minusPhrasesMap.getValue(it.minusKeywordsId) else listOf()
                val adGroupMinusPhrasesProcessQuoted = adGroupMinusPhrases
                    .map { minusPhrase -> processQuotedPhrase(minusPhrase) }
                AdGroupMinusPhrasesInfo(adGroupId = it.id, campaignId = it.campaignId, adGroupMinusPhrasesProcessQuoted)
            }
    }

    fun getResourceForAdGroupsWithLibraryMinusPhrases(adGroups: Collection<AdGroup>,
                                                      minusPhrasesMap: Map<Long, MinusPhrasesList>): List<AdGroupMinusPhrasesInfo> {
        val adGroupToMinusPhrases = adGroups
            .associate {
                val libraryMinusPhrases = it.libraryMinusKeywordsIds.flatMap { id ->
                    minusPhrasesMap.getOrDefault(id, listOf())
                }

                val adGroupMinusPhrases: List<String> =
                    if (it.minusKeywordsId != null && minusPhrasesMap.containsKey(it.minusKeywordsId))
                        minusPhrasesMap.getValue(it.minusKeywordsId) else listOf()

                val allMinusPhrases = (adGroupMinusPhrases + libraryMinusPhrases).distinct()
                AdGroupMinusPhrasesHandler.AdGroupIdWithCampaignId(it.id, it.campaignId) to allMinusPhrases
            }

        return Trace.current().profile("process_minus_phrases_list").use { _ ->
            val processIndividualPhrasesTask = Callable<Map<String, PhraseToNormForm>> {
                adGroupToMinusPhrases.values
                    .flatMapTo(hashSetOf()) { it }
                    .parallelStream()
                    .map { it to processMinusPhrase(it) }
                    .collect(Collectors.toMap({ pair -> pair.first!! }, { pair -> pair.second }))
            }

            val minusPhrasesForAllAdGroups: Map<String, PhraseToNormForm> =
                minusPhrasesProcessJoinPool.submit(processIndividualPhrasesTask).get()

            val formatResultTask = Callable<List<AdGroupMinusPhrasesInfo>> {
                adGroupToMinusPhrases
                    .entries
                    .parallelStream()
                    .map { (key, minusPhrases) ->
                        AdGroupMinusPhrasesInfo(
                            adGroupId = key.adGroupId,
                            campaignId = key.campaignId,
                            minusPhrases = processMinusPhrasesList(minusPhrases, minusPhrasesForAllAdGroups),
                        )
                    }
                    .collect(Collectors.toList())
            }

            minusPhrasesProcessJoinPool.submit(formatResultTask).get()
        }
    }

    private fun processMinusPhrasesList(
        groupMinusPhrases: List<String>,
        processedMinusPhrases: Map<String, PhraseToNormForm>,
    ): List<String> =
        groupMinusPhrases
            .asSequence()
            .map { minusPhrase -> processedMinusPhrases.getValue(minusPhrase) }
            .filterNot { phraseWithNormForm -> phraseWithNormForm.phrase.isEmpty() }
            .distinctBy { phraseWithNormForm ->
                if (phraseWithNormForm.normForm.isEmpty())
                    phraseWithNormForm.phrase
                else phraseWithNormForm.normForm
            }
            .map { phraseWithNormForm -> phraseWithNormForm.phrase }
            .sortedBy { minusPhrase -> minusPhrase.toLowerCase() }
            .mapTo(ArrayList()) { minusPhrase -> processQuotedPhrase(minusPhrase) }

    private fun processQuotedPhrase(phrase: String): String {
        if (!(phrase.startsWith("\"") && phrase.endsWith("\""))) {
            return phrase
        }
        // нельзя объединить со MarkStopWordsTransform, так как в перле процессинг фраз в кавычках идет после сортировки
        // всех минус фраз, нужно сохранить логику перла для переезда
        return phrase.removeSurrounding("\"").split(" ")
            .map { if (stopWordService.isStopWord(it)) "+$it" else it }
            .joinToString(" ") + " ~0"

    }


    private fun getForkJoinPoolFactory(): ForkJoinPool.ForkJoinWorkerThreadFactory {
        return ForkJoinPool.ForkJoinWorkerThreadFactory { pool ->
            val worker = ForkJoinPool.defaultForkJoinWorkerThreadFactory.newThread(pool)
            worker.isDaemon = true
            worker.name = "minus-phrases-process-" + worker.poolIndex
            worker
        }
    }

    override fun close() {
        minusPhrasesProcessJoinPool.shutdown()
    }
}

class MarkStopWordsTransform(private val stopWordService: StopWordService) : ExpressionStopWordsTransform() {
    override fun visitRawStopWord(expr: Word): Expression {
        return when {
            isInsideSquareBrackets || isInsideQuotedWords -> {
                expr
            }
            else -> {
                Word(WordKind.FIXED, expr.text)
            }
        }
    }

    override fun visitWord(expr: Word): Expression {
        return if (expr.kind == WordKind.RAW && stopWordService.isStopWord(expr.text, false)) {
            visitRawStopWord(expr)
        } else {
            expr
        }
    }
}

class RemovePeriodTransform : ExpressionTransform() {

    companion object {
        private val MORE_THAN_ONE_PERIODS_REGEX = "\\.{2,}".toRegex()
        private val NOT_ONLY_NUMBERS_AND_SPECIAL_SYMBOLS_REGEX = "[^${TextConstants.NUMBERS}.${SPECIAL_SYMBOLS}]".toRegex()
    }

    override fun visitWord(expr: Word): Expression {
        var text = expr.text.replace(MORE_THAN_ONE_PERIODS_REGEX, ".")
        text = if (text.contains(NOT_ONLY_NUMBERS_AND_SPECIAL_SYMBOLS_REGEX))
            text.replace(".", " ")
        else text
        return Word(expr.kind, text)
    }
}

data class PhraseToNormForm(
    val phrase: String,
    val normForm: String
)
