package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.AdGroupShowCondition
import ru.yandex.direct.bstransport.yt.repository.adgroup.showcondition.AdGroupShowConditionYtRepository
import ru.yandex.direct.bstransport.yt.utils.CaesarIterIdGenerator
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData
import ru.yandex.direct.common.log.service.LogBsExportEssService
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator
import java.time.Clock

@Component
class AdGroupShowConditionExporter(
    private val logBsExportEssService: LogBsExportEssService,
    private val caesarIterIdGenerator: CaesarIterIdGenerator,
    private val bsOrderIdCalculator: BsOrderIdCalculator,
    private val ytRepository: AdGroupShowConditionYtRepository,
    private val clock: Clock = Clock.systemUTC(),
) {

    companion object {
        private const val LOG_TYPE = "ad_group_show_condition"
    }

    fun export(shard: Int, exportObjects: List<AdGroupShowConditionExportInfo>) {
        if (exportObjects.isEmpty()) return

        val iterId = caesarIterIdGenerator.generateCaesarIterId()
        val updateTime = clock.instant().epochSecond

        val campaignIds: Set<Long> = exportObjects.mapNotNull { it.campaignId }.toSet()
        val campaignToOrderId: Map<Long, Long> = bsOrderIdCalculator.calculateOrderIdIfNotExist(shard, campaignIds)

        val exportAndLogInfoList = exportObjects.map {
            it.showConditionBuilder.iterId = iterId
            it.showConditionBuilder.updateTime = updateTime
            ExportAndLogInfo(
                showCondition = it.showConditionBuilder.build(),
                campaignId = it.campaignId,
                orderId = it.campaignId?.let { cid -> campaignToOrderId[cid] })
        }

        ytRepository.modify(exportAndLogInfoList.map { it.showCondition })
        logExportedObjects(exportAndLogInfoList)
    }

    private fun logExportedObjects(objects: List<ExportAndLogInfo>) {
        val logsData = objects.map {
            LogBsExportEssData<AdGroupShowCondition>()
                .withPid(it.showCondition.adGroupId)
                .withCid(it.campaignId ?: 0L)
                .withOrderId(it.orderId ?: 0L)
                .withData(it.showCondition)
        }
        logBsExportEssService.logData(logsData, LOG_TYPE)
    }
}

data class AdGroupShowConditionExportInfo(
    val showConditionBuilder: AdGroupShowCondition.Builder,
    val campaignId: Long?  // в случае удаления информации о кампании нет
)

private data class ExportAndLogInfo(
    val showCondition: AdGroupShowCondition,
    val campaignId: Long?,
    val orderId: Long?,
)
