package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.AdGroupShowCondition
import ru.yandex.adv.direct.adgroup.AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_GEO
import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.adv.direct.expression2.TargetingExpressionAtom
import ru.yandex.direct.core.bsexport.repository.adgroup.showconditions.AdGroupShowConditionsGeoInfo
import ru.yandex.direct.core.bsexport.repository.adgroup.showconditions.BsExportAdgroupShowConditionsGeoRepository
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupShowConditionType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupShowConditionObject
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExportInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExporter
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionHandler
import ru.yandex.direct.regions.GeoTreeFactory

@Component
class AdGroupShowConditionGeoHandler(
    private val adGroupRepository: AdGroupRepository,
    private val showConditionsGeoRepository: BsExportAdgroupShowConditionsGeoRepository,
    private val exporter: AdGroupShowConditionExporter,
    private val geoTreeFactory: GeoTreeFactory
) : AdGroupShowConditionHandler {
    override fun resourceType() = AdGroupShowConditionType.GEO

    private val MOON_REGION_ID: Long = 20001

    override fun handle(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>) {
        if (objects.isEmpty()) return

        val showConditionInfo = loadShowConditionInfo(shard, objects)

        val preExportInfos = showConditionInfo.map { (groupId, geoInfo) ->
            val condition = convertToCondition(geoInfo)
            val exportObjectBuilder = AdGroupShowCondition.newBuilder()
                .apply {
                    adGroupId = groupId
                    type = AD_GROUP_SHOW_CONDITION_TYPE_GEO.number
                    if (condition != null) {
                        this.condition = condition
                    }
                }
            AdGroupShowConditionExportInfo(exportObjectBuilder, geoInfo.campaignId)
        }
        exporter.export(shard, preExportInfos)
    }

    private fun loadShowConditionInfo(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>): Map<Long, AdGroupShowConditionsGeoInfo> {
        val (byAdGroup, byBanner) = objects.partition { it.adGroupId != null }
        val bannerIds = byBanner.map { it.bannerId }.distinct()
        val adGroupIds = byAdGroup.map { it.adGroupId!! }.distinct() +
            adGroupRepository.getAdGroupIdsByBannerIds(shard, bannerIds)
        return showConditionsGeoRepository.getGeoInfoByAdGroupIds(shard, adGroupIds)
    }

    private fun convertToCondition(geoInfo: AdGroupShowConditionsGeoInfo): TargetingExpression? {
        val expression = TargetingExpression.newBuilder()
        val geoTree = geoTreeFactory.apiGeoTree
        val newGeo = geoTree.excludeRegions(geoInfo.geo, geoInfo.minusGeo)
        if (newGeo.size == 1 && newGeo[0] == 0L) {
            return null
        }
        var (plusGeo, minusGeo) = newGeo.filter { it != 0L }.partition { it > 0 }
        if (newGeo.isEmpty()) {
            plusGeo = listOf(MOON_REGION_ID)
        }

        if (plusGeo.isNotEmpty()) {
            expression.addAnd(
                TargetingExpression.Disjunction.newBuilder().addAllOr(
                    plusGeo.sorted().map {
                        TargetingExpressionAtom.newBuilder()
                            .setKeyword(KeywordEnum.RegId.number)
                            .setOperation(OperationEnum.Equal.number)
                            .setValue(it.toString())
                            .build()
                    }
                ).build()
            )
        }
        minusGeo.sorted().forEach {
            expression.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    TargetingExpressionAtom.newBuilder()
                        .setKeyword(KeywordEnum.RegId.number)
                        .setOperation(OperationEnum.NotEqual.number)
                        .setValue((-it).toString())
                        .build()
                ).build()
            )
        }
        return expression.build()
    }
}
