package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.handler

import java.time.Duration
import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.AdGroupShowCondition
import ru.yandex.adv.direct.adgroup.AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_RF
import ru.yandex.adv.direct.adgroup.AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_TIME
import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.adv.direct.expression2.TargetingExpressionAtom
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupShowConditionType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupShowConditionObject
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler.AdGroupShowConditionsHandler
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExportInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExporter
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionHandler
import ru.yandex.direct.utils.DateTimeUtils.moscowDateTimeToEpochSecond

abstract class AdGroupShowConditionInternalHandler(
    private val adGroupRepository: AdGroupRepository,
    private val exporter: AdGroupShowConditionExporter,
) : AdGroupShowConditionHandler {
    abstract fun conditionType(): ru.yandex.adv.direct.adgroup.AdGroupShowConditionType

    abstract fun convertToCondition(adgroup: InternalAdGroup): TargetingExpression?

    override fun handle(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>) {
        if (objects.isEmpty()) return

        val showConditionInfo = loadShowConditionInfo(shard, objects)

        val preExportInfos = objects.map { obj ->
            val adGroup: InternalAdGroup? = showConditionInfo[obj.adGroupId]
            val exportObjectBuilder = AdGroupShowCondition.newBuilder()
                .apply {
                    adGroupId = obj.adGroupId!!
                    type = conditionType().number
                    if (adGroup != null) {
                        val cond = convertToCondition(adGroup)
                        if (cond != null) {
                            condition = cond
                        }
                    }
                }
            AdGroupShowConditionExportInfo(exportObjectBuilder, obj.campaignId)
        }
        exporter.export(shard, preExportInfos)
    }

    private fun loadShowConditionInfo(
        shard: Int,
        objects: Collection<BsExportAdGroupShowConditionObject>
    ): Map<Long, InternalAdGroup> {
        val adGroupIds = objects.map { it.adGroupId }.distinct()
        // Start/FinishTime/RF* есть только на группах внутренней рекламы
        return adGroupRepository.getAdGroups(shard, adGroupIds)
            .filterIsInstance(InternalAdGroup::class.java)
            .map { it.id to it }
            .toMap()
    }
}

@Component
class AdGroupShowConditionTimeHandler(
    private val adGroupRepository: AdGroupRepository,
    private val exporter: AdGroupShowConditionExporter,
) : AdGroupShowConditionInternalHandler(adGroupRepository, exporter) {
    override fun resourceType() = AdGroupShowConditionType.TIME

    override fun conditionType() = AD_GROUP_SHOW_CONDITION_TYPE_TIME

    override fun convertToCondition(adgroup: InternalAdGroup): TargetingExpression? {
        val startUnixTime = adgroup.startTime?.let { moscowDateTimeToEpochSecond(it) }
        val finishUnixTime = adgroup.finishTime?.let { moscowDateTimeToEpochSecond(it) }
        var hasValue = false

        val expression = TargetingExpression.newBuilder()
        if (startUnixTime != null) {
            expression.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    TargetingExpressionAtom.newBuilder()
                        .setKeyword(KeywordEnum.Unixtime.number)
                        .setOperation(OperationEnum.GreaterOrEqual.number)
                        .setValue(startUnixTime.toString())
                        .build()
                ).build()
            )
            hasValue = true
        }
        if (finishUnixTime != null) {
            expression.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    TargetingExpressionAtom.newBuilder()
                        .setKeyword(KeywordEnum.Unixtime.number)
                        .setOperation(OperationEnum.Less.number)
                        .setValue(finishUnixTime.toString())
                        .build()
                ).build()
            )
            hasValue = true
        }
        return if (hasValue) {
            expression.build()
        } else {
            null
        }
    }
}

@Component
class AdGroupShowConditionRFHandler(
    private val adGroupRepository: AdGroupRepository,
    private val exporter: AdGroupShowConditionExporter,
    private val ppcPropertiesSupport: PpcPropertiesSupport
) : AdGroupShowConditionInternalHandler(adGroupRepository, exporter) {

    private val oldRfExportByFilterEnabledProp = ppcPropertiesSupport.get(
        PpcPropertyNames.BS_EXPORT_OLD_RF_FILTER_INTERNAL_AD_GROUPS_ENABLED,
        Duration.ofMinutes(5),
    )
    private val filterByCampaignIdsProp = ppcPropertiesSupport.get(
        PpcPropertyNames.BS_EXPORT_OLD_RF_FILTER_BY_CAMPAIGN_IDS,
        Duration.ofMinutes(5),
    )

    override fun resourceType() = AdGroupShowConditionType.RF

    override fun conditionType() = AD_GROUP_SHOW_CONDITION_TYPE_RF

    override fun convertToCondition(adgroup: InternalAdGroup): TargetingExpression? {
        if (adgroup.rf == null || adgroup.rf == 0 || !AdGroupShowConditionsHandler.needExportGroupRfCondition(
                oldRfExportByFilterEnabledProp, filterByCampaignIdsProp, adgroup.campaignId)) {
            return null
        }
        val rfReset: Int = if (adgroup.rfReset != null) adgroup.rfReset else 0
        val expression = TargetingExpression.newBuilder()
        expression.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                TargetingExpressionAtom.newBuilder()
                    .setKeyword(KeywordEnum.FrequencyDay.number)
                    .setOperation(OperationEnum.Less.number)
                    .setValue(adgroup.rf.toString())
                    .build()
            ).addOr(
                TargetingExpressionAtom.newBuilder()
                    .setKeyword(KeywordEnum.FreqExpire.number)
                    .setOperation(OperationEnum.GreaterOrEqual.number)
                    .setValue(rfReset.toString())
                    .build()
            ).build()
        )
        return expression.build()
    }
}
