package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.AdGroupShowCondition
import ru.yandex.adv.direct.adgroup.AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_MOBILE_CONTENT
import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.adv.direct.expression2.TargetingExpressionAtom
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupDeviceTypeTargeting
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupNetworkTargeting
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.mobilecontent.model.OsType
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupShowConditionType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupShowConditionObject
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler.MobileAppsCommon.getMobileOsMinVersion
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.resource.handler.MobileAppsCommon.mobileAppsTargetingToStr
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExportInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExporter
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionHandler

@Component
class AdGroupShowConditionMobileContentHandler(
    private val adGroupRepository: AdGroupRepository,
    private val exporter: AdGroupShowConditionExporter,
    private val mobileContentRepository: MobileContentRepository,
) : AdGroupShowConditionHandler {

    override fun resourceType() = AdGroupShowConditionType.MOBILE_CONTENT

    override fun handle(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>) {
        if (objects.isEmpty()) return

        val showConditionInfo = loadShowConditionInfo(shard, objects)

        val preExportInfos = showConditionInfo.map { adGroup ->
            val exportObjectBuilder = AdGroupShowCondition.newBuilder()
                .apply {
                    adGroupId = adGroup.id
                    type = AD_GROUP_SHOW_CONDITION_TYPE_MOBILE_CONTENT.number
                    condition = convertToCondition(adGroup)
                }
            AdGroupShowConditionExportInfo(exportObjectBuilder, adGroup.campaignId)
        }
        exporter.export(shard, preExportInfos)
    }

    private fun loadShowConditionInfo(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>)
        : List<MobileContentAdGroup> {
        val (byAdGroup, byMobile) = objects.partition { it.adGroupId != null }
        val adGroupIds = byAdGroup.map { it.adGroupId }.distinct()
        val mobileContentIds = byMobile.map { it.mobileContentId }.distinct()
        val mobileAdGroupIds = mobileContentRepository.getAdgroupIdsForMobileContentIds(shard, mobileContentIds)
        return adGroupRepository.getAdGroups(shard, adGroupIds + mobileAdGroupIds)
            .filterIsInstance<MobileContentAdGroup>()
    }

    private fun addOsTypeCondition(adGroup: MobileContentAdGroup, expression: TargetingExpression.Builder) {
        val osTypeToInt = mapOf(
            OsType.ANDROID to "2",
            OsType.IOS to "3",
        )

        expression.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                TargetingExpressionAtom.newBuilder()
                    .setKeyword(KeywordEnum.DetailedDeviceType.number)
                    .setOperation(OperationEnum.Equal.number)
                    .setValue(osTypeToInt[adGroup.mobileContent.osType]!!)
                    .build()
            ).build()
        )
    }

    private fun addContentStoreCondition(adGroup: MobileContentAdGroup, expression: TargetingExpression.Builder) {
        expression.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                TargetingExpressionAtom.newBuilder()
                    .setKeyword(KeywordEnum.ExceptAppsOnCpi.number)
                    .setOperation(OperationEnum.NotEqual.number)
                    .setValue(mobileAppsTargetingToStr(adGroup.mobileContent))
                    .build()
            ).build()
        )
    }

    private fun addCellCondition(adGroup: MobileContentAdGroup, expression: TargetingExpression.Builder) {
        if (!adGroup.networkTargeting.contains(MobileContentAdGroupNetworkTargeting.CELLULAR)) {
            expression.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    TargetingExpressionAtom.newBuilder()
                        .setKeyword(KeywordEnum.StationaryConnection.number)
                        .setOperation(OperationEnum.Match.number)
                        .setValue("1")
                        .build()
                ).build()
            )
        }
    }

    private fun addDeviceTypeCondition(adGroup: MobileContentAdGroup, expression: TargetingExpression.Builder) {
        if (adGroup.deviceTypeTargeting.size == 0) {
            return
        }
        val targetingToInt = mapOf(
            MobileContentAdGroupDeviceTypeTargeting.PHONE to "3",
            MobileContentAdGroupDeviceTypeTargeting.TABLET to "4",
        )
        val builder = TargetingExpression.Disjunction.newBuilder()
        for (targeting in adGroup.deviceTypeTargeting) {
            builder.addOr(
                TargetingExpressionAtom.newBuilder()
                    .setKeyword(KeywordEnum.DeviceType.number)
                    .setOperation(OperationEnum.Equal.number)
                    .setValue(targetingToInt[targeting]!!)
                    .build()
            )
        }
        expression.addAnd(builder.build())
    }

    private fun addMobileOsMinVersionCondition(adGroup: MobileContentAdGroup, expression: TargetingExpression.Builder) {
        expression.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                TargetingExpressionAtom.newBuilder()
                    .setKeyword(KeywordEnum.OsMinVersion.number)
                    .setOperation(OperationEnum.GreaterOrEqual.number)
                    .setValue(getMobileOsMinVersion(adGroup).toString())
                    .build()
            ).build()
        )
    }

    private fun convertToCondition(adGroup: MobileContentAdGroup): TargetingExpression {
        val expression = TargetingExpression.newBuilder()

        addOsTypeCondition(adGroup, expression)
        addContentStoreCondition(adGroup, expression)
        addMobileOsMinVersionCondition(adGroup, expression)
        addCellCondition(adGroup, expression)
        addDeviceTypeCondition(adGroup, expression)

        return expression.build()
    }
}
