package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.AdGroupShowCondition
import ru.yandex.adv.direct.adgroup.AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_PLACE
import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.adv.direct.expression2.TargetingExpressionAtom
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupShowConditionType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupShowConditionObject
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExportInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExporter
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionHandler
import ru.yandex.grut.objects.proto.PlaceIds.EPlaceId.PI_DIRECT
import ru.yandex.grut.objects.proto.PlaceIds.EPlaceId.PI_REACH

@Component
class AdGroupShowConditionPlaceHandler(
    private val adGroupRepository: AdGroupRepository,
    private val campaignRepository: CampaignRepository,
    private val exporter: AdGroupShowConditionExporter,
) : AdGroupShowConditionHandler {

    override fun resourceType() = AdGroupShowConditionType.PLACE

    private val directPlaceId: Long = PI_DIRECT.number.toLong()
    private val reachDirectPlaceId: Long = PI_REACH.number.toLong()

    override fun handle(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>) {
        if (objects.isEmpty()) return

        val campaignTypesIdMap = loadCampaignsTypes(shard, objects)
        val (internal, external) = objects
            .filter { it.campaignId in campaignTypesIdMap }
            .partition { campaignTypesIdMap[it.campaignId] in CampaignTypeKinds.INTERNAL }
        val preExportInfos = mutableListOf<AdGroupShowConditionExportInfo>()

        handleExternal(shard, external, preExportInfos, campaignTypesIdMap)
        handleInternal(shard, internal, preExportInfos)

        exporter.export(shard, preExportInfos)
    }

    private fun handleExternal(shard: Int,
                               objects: Collection<BsExportAdGroupShowConditionObject>,
                               preExportInfos: MutableList<AdGroupShowConditionExportInfo>,
                               campaignTypes: Map<Long, CampaignType>) {
        objects.forEach {
            val placeId = if (campaignTypes[it.campaignId] in CampaignTypeKinds.CPM) reachDirectPlaceId else directPlaceId
            val builder = AdGroupShowCondition.newBuilder()
                .setAdGroupId(it.adGroupId!!)
                .setType(AD_GROUP_SHOW_CONDITION_TYPE_PLACE.number)
                .setCondition(convertToCondition(placeId))
            preExportInfos.add(AdGroupShowConditionExportInfo(builder, it.campaignId))
        }
    }

    private fun handleInternal(shard: Int,
                               objects: Collection<BsExportAdGroupShowConditionObject>,
                               preExportInfos: MutableList<AdGroupShowConditionExportInfo>) {
        val placesByAdGroups = loadPlacesByAdGroups(shard, objects)

        placesByAdGroups.forEach { (adGroupId, placeId) ->
            val builder = AdGroupShowCondition.newBuilder()
                .setAdGroupId(adGroupId)
                .setType(AD_GROUP_SHOW_CONDITION_TYPE_PLACE.number)
                .setCondition(convertToCondition(placeId))
            preExportInfos.add(AdGroupShowConditionExportInfo(builder, null))
        }
    }

    private fun loadCampaignsTypes(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>)
    : Map<Long, CampaignType> {
        val campaignIds = objects.mapNotNull { it.campaignId }.distinct()
        return campaignRepository.getCampaignsTypeMap(shard, campaignIds)
    }

    private fun loadPlacesByAdGroups(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>)
    : Map<Long, Long> {
        val (byAdGroupId, byCampaignId) = objects.partition { it.adGroupId != null }
        val adGroupIds = byAdGroupId.map { it.adGroupId }.distinct()
        val campaignIds = byCampaignId.map { it.campaignId }.distinct()

        val adGroupsByCampaigns = adGroupRepository.getAdGroupIdsByCampaignIds(shard, campaignIds)
        val placesByAdGroups = mutableMapOf<Long, Long>()
        placesByAdGroups += campaignRepository.getCampaignInternalPlacesByAdGroupIds(shard, adGroupIds)
        val placesByCampaigns = campaignRepository.getCampaignInternalPlaces(shard, campaignIds)

        for ((campaignId, adGroups) in adGroupsByCampaigns) {
            adGroups.forEach {
                placesByAdGroups[it] = placesByCampaigns[campaignId]!!
            }
        }
        return placesByAdGroups
    }

    private fun convertToCondition(placeId: Long): TargetingExpression {
        return TargetingExpression.newBuilder().addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                TargetingExpressionAtom.newBuilder()
                    .setKeyword(KeywordEnum.PlaceId.number)
                    .setOperation(OperationEnum.Equal.number)
                    .setValue(placeId.toString())
                    .build()
            )
        ).build()
    }
}
