package ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.adgroup.AdGroupShowCondition
import ru.yandex.adv.direct.adgroup.AdGroupShowConditionType.AD_GROUP_SHOW_CONDITION_TYPE_TARGET_TAGS
import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.adv.direct.expression2.TargetingExpressionAtom
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupShowConditionType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupShowConditionObject
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.common.getTargetTags
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExportInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionExporter
import ru.yandex.direct.logicprocessor.processors.bsexport.adgroup.showcondition.AdGroupShowConditionHandler

@Component
class AdGroupShowConditionTargetTagsHandler(
    private val adGroupRepository: AdGroupRepository,
    private val exporter: AdGroupShowConditionExporter
): AdGroupShowConditionHandler {

    override fun resourceType() = AdGroupShowConditionType.TARGET_TAGS

    override fun handle(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>) {
        if (objects.isEmpty()) return

        val showConditionInfo = loadShowConditionInfo(shard, objects)

        val preExportInfos = objects.map { obj ->
            val tags = showConditionInfo[obj.adGroupId]
            val exportObjectBuilder = AdGroupShowCondition.newBuilder()
                .setAdGroupId(obj.adGroupId!!)
                .setType(AD_GROUP_SHOW_CONDITION_TYPE_TARGET_TAGS.number)

            if (tags != null) {
                convertToCondition(tags)?.let { exportObjectBuilder.setCondition(it) }
            }

            AdGroupShowConditionExportInfo(exportObjectBuilder, obj.campaignId)
        }
        exporter.export(shard, preExportInfos)
    }

    private fun loadShowConditionInfo(shard: Int, objects: Collection<BsExportAdGroupShowConditionObject>)
    : Map<Long, List<String>?> {
        val adGroupIds = objects.map { it.adGroupId }.distinct()
        val adGroups = adGroupRepository.getAdGroups(shard, adGroupIds)
        return adGroups.map {
            it.id to getTargetTags(it)
        }.toMap()
    }

    private fun convertToCondition(tags: List<String>): TargetingExpression? {
        if (tags.isEmpty())
            return null
        val expression = TargetingExpression.newBuilder()

        expression.addAnd(
            TargetingExpression.Disjunction.newBuilder().addAllOr(
                tags.map { tag ->
                    TargetingExpressionAtom.newBuilder()
                        .setKeyword(KeywordEnum.TargetTags.number)
                        .setOperation(OperationEnum.Match.number)
                        .setValue(tag)
                        .build()
                }
            ).build()
        )
        return expression.build()
    }
}
