package ru.yandex.direct.logicprocessor.processors.bsexport.bids

import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.bsexport.model.BsExportAbstractBid
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsDynamicRepository
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsKeywordRepository
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsOfferRetargetingRepository
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsPerformanceRepository
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsRelevanceMatchRepository
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsRepositoryInterface
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsRetargetingRepository
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsWithParamsRepositoryInterface
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BidObjectType
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BsExportBidsObject
import java.time.Duration

@Component
class BidRepositoryDispatcher(
    bidsPerformanceRepository: BsExportBidsPerformanceRepository,
    bidsOfferRetargetingRepository: BsExportBidsOfferRetargetingRepository,
    private val bidRepository: BsExportBidsKeywordRepository,
    private val bidsDynamicRepository: BsExportBidsDynamicRepository,
    private val bidsRetargetingRepository: BsExportBidsRetargetingRepository,
    private val bidsRelevanceMatchRepository: BsExportBidsRelevanceMatchRepository,
    private val monitoring: BidRetargetingConditionMonitoring,

    ppcPropertiesSupport: PpcPropertiesSupport,
) {
    val bidRepositories: Map<BidObjectType, BsExportBidsRepositoryInterface<*>> = mapOf(
        BidObjectType.PERFORMANCE to bidsPerformanceRepository,
        BidObjectType.OFFER_RETARGETING to bidsOfferRetargetingRepository,
        BidObjectType.KEYWORD to bidRepository,
        BidObjectType.DYNAMIC to bidsDynamicRepository,
        BidObjectType.RETARGETING to bidsRetargetingRepository,
        BidObjectType.RELEVANCE_MATCH to bidsRelevanceMatchRepository,
    )

    fun getBids(
        shard: Int,
        objects: Map<BidObjectType, Collection<BsExportBidsObject>>,
    ): Sequence<BsExportAbstractBid> =
        objects
            .asSequence()
            .flatMap { (type, objectsOfType) -> dispatch(shard, type, objectsOfType) }

    private fun dispatch(
        shard: Int,
        type: BidObjectType,
        objects: Collection<BsExportBidsObject>,
    ): Sequence<BsExportAbstractBid> = when (type) {
        BidObjectType.BIDS_HREF_PARAMS -> sequence {
            val keys =
                objects.map { BsExportBidsWithParamsRepositoryInterface.HrefParamKey(it.id, checkNotNull(it.cid)) }
            yieldAll(bidRepository.getBidsByHrefParamsLazily(shard, keys))
            yieldAll(bidsRelevanceMatchRepository.getBidsByHrefParamsLazily(shard, keys))
        }
        BidObjectType.RETARGETING_CONDITION -> when {
            isRetargetingConditionIgnored -> emptySequence()
            else -> {
                bidsRetargetingRepository
                    .getBidsByRetargetingConditions(shard, objects.map { it.id })
                    .onEach { monitoring.updateRetargetingConditionBidsCount(shard, it.size.toLong()) }
                    .flatten()
            }
        }
        BidObjectType.DYNAMIC_CONDITION -> {
            bidsDynamicRepository
                .getBidsByDynamicConditions(shard, objects.map { it.id })
                .asSequence()
        }
        else -> {
            bidRepositories
                .getValue(type)
                .getBids(shard, objects.map { it.id })
                .asSequence()
        }
    }

    private val retargetingConditionIgnoredProperty = ppcPropertiesSupport.get(
        PpcPropertyNames.BS_EXPORT_ESS_BIDDABLE_SHOWS_CONDITIONS_RET_COND_IDS_IGNORED,
        Duration.ofMinutes(5)
    )

    private val isRetargetingConditionIgnored: Boolean
        get() = retargetingConditionIgnoredProperty.get() ?: false
}
