package ru.yandex.direct.logicprocessor.processors.bsexport.bids

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.adv.direct.showcondition.DynamicConditionData
import ru.yandex.adv.direct.showcondition.FeedFilterData
import ru.yandex.adv.direct.showcondition.HrefParam
import ru.yandex.adv.direct.showcondition.KeywordData
import ru.yandex.adv.direct.showcondition.RetargetingData
import ru.yandex.adv.direct.showconditions.BiddableShowCondition
import ru.yandex.direct.bstransport.yt.utils.CaesarIterIdGenerator
import ru.yandex.direct.bstransport.yt.utils.ToBsConversionUtils.bsMoneyForPerformance
import ru.yandex.direct.bstransport.yt.utils.ToBsConversionUtils.bsPreparePrice
import ru.yandex.direct.common.configuration.CommonConfiguration.SYSTEM_UTC_CLOCK_NAME
import ru.yandex.direct.core.bsexport.model.BidsStatusModerate
import ru.yandex.direct.core.bsexport.model.BsExportAbstractBid
import ru.yandex.direct.core.bsexport.model.BsExportBidDynamic
import ru.yandex.direct.core.bsexport.model.BsExportBidKeyword
import ru.yandex.direct.core.bsexport.model.BsExportBidOfferRetargeting
import ru.yandex.direct.core.bsexport.model.BsExportBidPerformance
import ru.yandex.direct.core.bsexport.model.BsExportBidRetargeting
import ru.yandex.direct.core.bsexport.model.BsExportRelevanceMatch
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.CPA_STRATEGIES
import ru.yandex.direct.core.entity.client.service.ClientNdsService
import ru.yandex.direct.core.entity.keyword.processing.bsexport.BsExportTextProcessor
import ru.yandex.direct.core.entity.performancefilter.model.TargetFunnel
import ru.yandex.direct.currency.Percent
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BidObjectType
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.YtRecordMappingUtils.relevanceMatchCategoriesToYt
import ru.yandex.direct.model.Model
import ru.yandex.direct.utils.CommonUtils
import java.time.Clock

data class BiddableShowConditionWithCid(
    val bid: BiddableShowCondition,
    val cid: Long
)

@Component
open class BiddableShowConditionsYtRecordMapper(
    private val bsExportTextProcessor: BsExportTextProcessor,
    private val clientNdsService: ClientNdsService,
    private val caesarIterIdGenerator: CaesarIterIdGenerator,
    @Qualifier(SYSTEM_UTC_CLOCK_NAME)
    private val clock: Clock
) {
    data class Now(val iterId: Long, val time: Long)

    companion object {
        // https://a.yandex-team.ru/arc/trunk/arcadia/yabs/server/cs/libs/direct_object/context_bids.cpp?rev=6910787#L5
        // https://a.yandex-team.ru/arc/trunk/arcadia/yabs/server/libs/enums/context_type.h?rev=6910787
        private val CONTEXT_TYPE: Map<Class<out Model>, Int> = java.util.Map.of(
            BsExportBidKeyword::class.java, 1,  // PHRASE/Normal
            BsExportBidRetargeting::class.java, 2,  // GOAL_CONTEXT/Goal
            BsExportBidDynamic::class.java, 7,  // DYNAMIC/CompactBroad
            BsExportBidPerformance::class.java, 8,  // FILTER/Offer
            BsExportBidOfferRetargeting::class.java, 8,  //OfferRetargeting for smart banners in text group
            BsExportRelevanceMatch::class.java, 11 // RELEVANCE_MATCH/RelevanceMatch
        )
        private val OBJECT_TYPES: Map<BidObjectType?, Class<out Model>> = java.util.Map.of(
            BidObjectType.KEYWORD, BsExportBidKeyword::class.java,
            BidObjectType.RETARGETING, BsExportBidRetargeting::class.java,
            BidObjectType.DYNAMIC, BsExportBidDynamic::class.java,
            BidObjectType.PERFORMANCE, BsExportBidPerformance::class.java,
            BidObjectType.OFFER_RETARGETING, BsExportBidOfferRetargeting::class.java,
            BidObjectType.RELEVANCE_MATCH, BsExportRelevanceMatch::class.java
        )

        @JvmStatic
        fun <T : Model> contextTypeOf(bid: T): Int =
            CONTEXT_TYPE[bid::class.java]!!

        @JvmStatic
        fun contextTypeOf(bidObjectType: BidObjectType): Int =
            CONTEXT_TYPE[OBJECT_TYPES[bidObjectType]]!!
    }

    fun bidsToYtRecords(bids: List<BsExportAbstractBid>, now: Now): List<BiddableShowConditionWithCid> {
        val ndsInfo = getNdsInfoForPerformanceBids(bids)

        return bids.map {
            val bid = when (it) {
                is BsExportBidKeyword -> keywordBidToYt(it, now)
                is BsExportRelevanceMatch -> relevanceMatchBidToYt(it, now)
                is BsExportBidRetargeting -> retargetingBidToYt(it, now)
                is BsExportBidPerformance -> performanceBidToYt(it, now, ndsInfo[it.clientId])
                is BsExportBidOfferRetargeting -> offerRetargetingBidToYt(it, now, ndsInfo[it.clientId])
                is BsExportBidDynamic -> dynamicBidToYt(it, now)
                else -> throw IllegalStateException("Unsupported type: ${it::class.qualifiedName}")
            }
            BiddableShowConditionWithCid(bid, it.campaignId)
        }
    }

    private fun getNdsInfoForPerformanceBids(bids: List<BsExportAbstractBid>): Map<Long, Percent> {
        val perfBids = bids.filterIsInstance<BsExportBidPerformance>()
        return if (perfBids.isEmpty()) {
            mapOf()
        } else {
            val clientIds = perfBids.map { it.clientId }.filterNotNull().map { ClientId.fromLong(it) }.distinct()
            clientNdsService.massGetEffectiveClientNdsByIds(clientIds).associate { it.clientId to it.nds }
        }
    }

    fun keywordBidToYt(kw: BsExportBidKeyword, now: Now): BiddableShowCondition {
        val bidValue = bsPreparePrice(kw.price, kw.currency, kw.campaignType)
        val bidContextValue = bsPreparePrice(kw.priceContext, kw.currency, kw.campaignType)
        val keywordBuilder = KeywordData.newBuilder()
            .setText(getKeywordText(kw))
        if (kw.showsForecast != null) {
            keywordBuilder.hits = kw.showsForecast
        }
        if (kw.calculatedPhraseId != null) {
            keywordBuilder.phraseId = kw.calculatedPhraseId.toLong()
        }
        return BiddableShowCondition.newBuilder()
            .setIterId(now.iterId)
            .setOrderId(kw.orderId)
            .setContextType(contextTypeOf(kw))
            .setAdGroupId(kw.adGroupId)
            .setId(kw.id)
            .setBid(bidValue)
            .setBidContext(bidContextValue)
            .setKeywordData(keywordBuilder.build())
            .addAllHrefParams(kw.params.toHrefParams())
            .setSuspended(kw.isSuspended)
            .setUpdateTime(now.time)
            .setDeleteTime(if (kw.statusModerate == BidsStatusModerate.YES) 0 else now.time)
            .build()
    }

    fun getKeywordText(kw: BsExportBidKeyword) =
        bsExportTextProcessor.processQuotedText(kw.phrase)

    fun dynamicBidToYt(dynamic: BsExportBidDynamic, now: Now): BiddableShowCondition {
        val bidValue = bsPreparePrice(dynamic.price, dynamic.currency, dynamic.campaignType)
        val bidContextValue = bsPreparePrice(dynamic.priceContext, dynamic.currency, dynamic.campaignType)
        val data = DynamicConditionData.newBuilder()
            .setDynCondId(dynamic.dynCondId)
            .build()
        return BiddableShowCondition.newBuilder()
            .setIterId(now.iterId)
            .setOrderId(dynamic.orderId)
            .setContextType(contextTypeOf(dynamic))
            .setAdGroupId(dynamic.adGroupId)
            .setId(dynamic.id)
            .setBid(bidValue)
            .setBidContext(bidContextValue)
            .setDynamicConditionData(data)
            .addAllHrefParams(dynamic.params.toHrefParams())
            .setSuspended(dynamic.isSuspended)
            .setUpdateTime(now.time)
            .setDeleteTime(0)
            .build()
    }

    fun performanceBidToYt(perfBid: BsExportBidPerformance, now: Now, nds: Percent?): BiddableShowCondition {
        val bidCpc = bsMoneyForPerformance(
            perfBid.priceCpc, perfBid.campaignStrategyData.filterAvgBid,
            perfBid.currency, nds
        )
        val bidCpa =
            if (perfBid.campaignStrategyName != null && CPA_STRATEGIES.contains(perfBid.campaignStrategyName))
                bsMoneyForPerformance(
                    perfBid.priceCpa, perfBid.campaignStrategyData.filterAvgCpa,
                    perfBid.currency, nds
                )
            else 0
        val data = FeedFilterData.newBuilder()
            .setOnlyOfferRetargeting(perfBid.targetFunnel == TargetFunnel.PRODUCT_PAGE_VISIT)
            .setOnlyNewAuditory(perfBid.targetFunnel == TargetFunnel.NEW_AUDITORY)
            .setPriceCpc(bidCpc)
            .setPriceCpa(bidCpa)
            .build()
        val builder = BiddableShowCondition.newBuilder()
            .setIterId(now.iterId)
            .setOrderId(perfBid.orderId)
            .setContextType(contextTypeOf(perfBid))
            .setAdGroupId(perfBid.adGroupId)
            .setId(perfBid.id)
            .setFeedFilterData(data)
            .addAllHrefParams(perfBid.params.toHrefParams())
            .setSuspended(perfBid.isSuspended)
            .setUpdateTime(now.time)
        if (CommonUtils.nvl(perfBid.isDeleted, false)) {
            builder.deleteTime = now.time
        } else {
            builder.deleteTime = 0
        }
        return builder.build()
    }

    /**
     * Конвертация офферного ретаргетинга в YT представление
     * отличается от performanceBidToYt тем, что ставка берется всегда из стратегии кампании и не заполняются поля
     * onlyNewAuditory и onlyOfferRetargeting: https://st.yandex-team.ru/DIRECT-161094#620f76d4e1bb6d54865e64d0
     */
    fun offerRetargetingBidToYt(
        offerRetargeting: BsExportBidOfferRetargeting,
        now: Now,
        nds: Percent?
    ): BiddableShowCondition {
        val bidCpc = bsMoneyForPerformance(
            null, offerRetargeting.campaignStrategyData.filterAvgBid,
            offerRetargeting.currency, nds
        )
        val bidCpa =
            if (offerRetargeting.campaignStrategyName != null
                && CPA_STRATEGIES.contains(offerRetargeting.campaignStrategyName)
            )
                bsMoneyForPerformance(
                    null, offerRetargeting.campaignStrategyData.filterAvgCpa,
                    offerRetargeting.currency, nds
                )
            else 0
        val deleteTime =
            if (offerRetargeting.isDeleted == true) now.time
            else 0

        val data = FeedFilterData.newBuilder()
            .setPriceCpc(bidCpc)
            .setPriceCpa(bidCpa)
            .build()
        return BiddableShowCondition.newBuilder()
            .setIterId(now.iterId)
            .setOrderId(offerRetargeting.orderId)
            .setContextType(contextTypeOf(offerRetargeting))
            .setAdGroupId(offerRetargeting.adGroupId)
            .setId(offerRetargeting.id)
            .setFeedFilterData(data)
            .addAllHrefParams(offerRetargeting.params.toHrefParams())
            .setSuspended(offerRetargeting.isSuspended)
            .setUpdateTime(now.time)
            .setDeleteTime(deleteTime)
            .build()
    }

    fun retargetingBidToYt(ret: BsExportBidRetargeting, now: Now): BiddableShowCondition {
        val bidContextValue = bsPreparePrice(ret.priceContext, ret.currency, ret.campaignType)
        val data = RetargetingData.newBuilder()
            .setRetCondId(ret.retCondId)
            .build()
        return BiddableShowCondition.newBuilder()
            .setIterId(now.iterId)
            .setOrderId(ret.orderId)
            .setContextType(contextTypeOf(ret))
            .setAdGroupId(ret.adGroupId)
            .setId(ret.id)
            .setBidContext(bidContextValue)
            .setRetargetingData(data)
            .addAllHrefParams(ret.params.toHrefParams())
            .setSuspended(ret.isSuspended)
            .setUpdateTime(now.time)
            .setDeleteTime(if (ret.isAccessible) 0 else now.time)
            .build()
    }

    fun relevanceMatchBidToYt(rm: BsExportRelevanceMatch, now: Now): BiddableShowCondition {
        val bidValue = bsPreparePrice(rm.price, rm.currency, rm.campaignType)
        val bidContextValue = bsPreparePrice(rm.priceContext, rm.currency, rm.campaignType)
        val data = relevanceMatchCategoriesToYt(rm.relevanceMatchCategories)
        return BiddableShowCondition.newBuilder()
            .setIterId(now.iterId)
            .setOrderId(rm.orderId)
            .setAdGroupId(rm.adGroupId)
            .setContextType(contextTypeOf(rm))
            .setId(rm.id)
            .setBid(bidValue)
            .setBidContext(bidContextValue)
            .setRelevanceMatchData(data)
            .addAllHrefParams(rm.params.toHrefParams())
            .setUpdateTime(now.time)
            .setDeleteTime(if (rm.isDeleted ?: false) now.time else 0)
            .setSuspended(rm.isSuspended)
            .build()
    }

    fun createDeleteItem(
        orderId: Long, pid: Long, id: Long, bidObjectType: BidObjectType, now: Now
    ): BiddableShowCondition {
        return BiddableShowCondition.newBuilder()
            .setIterId(now.iterId)
            .setOrderId(orderId)
            .setAdGroupId(pid)
            .setContextType(contextTypeOf(bidObjectType))
            .setId(id)
            .setDeleteTime(now.time)
            .setUpdateTime(now.time)
            .build()
    }

    fun now() =
        Now(caesarIterIdGenerator.generateCaesarIterId(), clock.instant().epochSecond)
}

private fun Map<Int, String?>.toHrefParams(): List<HrefParam> {
    val nonNull = this.filterValues { !it.isNullOrEmpty() }
    return nonNull.map { (k, v) ->
        HrefParam.newBuilder().run {
            name = "" // имя не читается в Цезаре
            value = v
            paramNo = k
            build()
        }
    }
}
