package ru.yandex.direct.logicprocessor.processors.bsexport.bids

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.adv.direct.showconditions.BiddableShowCondition
import ru.yandex.direct.bstransport.yt.repository.showconditions.BiddableShowConditionYtRepository
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData
import ru.yandex.direct.common.log.service.LogBsExportEssService
import ru.yandex.direct.core.bsexport.model.BsExportAbstractBid
import ru.yandex.direct.core.bsexport.model.BsExportBidKeyword
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.bs.PhraseIdByDataMd5Repository
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository
import ru.yandex.direct.core.entity.statistics.container.ChangedPhraseIdInfo
import ru.yandex.direct.core.entity.statistics.repository.BsAuctionStatRepository
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BidObjectType
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BsExportBidsObject
import ru.yandex.direct.utils.HashingUtils
import java.math.BigInteger
import java.time.Duration

@Component
class BsExportBiddableShowConditionsService(
    private val ytRepository: BiddableShowConditionYtRepository,
    private val mapper: BiddableShowConditionsYtRecordMapper,
    private val bsOrderIdCalculator: BsOrderIdCalculator,
    private val adGroupRepository: AdGroupRepository,
    private val logBsExportEssService: LogBsExportEssService,
    private val bidRepositoryDispatcher: BidRepositoryDispatcher,
    private val historicalPhraseIdRepository: PhraseIdByDataMd5Repository,
    private val bsAuctionStatRepository: BsAuctionStatRepository,
    private val keywordRepository: KeywordRepository,

    ppcPropertiesSupport: PpcPropertiesSupport,
) {
    private val logDisabledProp = ppcPropertiesSupport.get(
        PpcPropertyNames.BS_EXPORT_ESS_BIDDABLE_SHOWS_CONDITIONS_LOG_DISABLED,
        Duration.ofMinutes(5),
    )

    fun updateResources(shard: Int, logicObjects: List<BsExportBidsObject>) {
        logger.debug("Processing objects for (shard: {}): {}", shard, logicObjects)
        val (updatedObjects, deletedObjects) = splitObjects(logicObjects)

        val bids = bidRepositoryDispatcher
            .getBids(shard, updatedObjects)
            .toList()

        calculatePhraseIds(bids)

        // прописываем недостающие OrderID
        fillMissedOrderId(shard, bids)

        val now = mapper.now()
        val ytUpdates = mapper.bidsToYtRecords(bids, now) +
            convertToDeletes(shard, mapper, deletedObjects, now)

        updateApplicableShowConditions(ytUpdates)
        saveChangedPhraseIds(shard, bids)
    }

    private fun calculatePhraseIds(bids: List<BsExportAbstractBid>) {
        val keywordBids = bids.filterIsInstance<BsExportBidKeyword>()
            .onEach {
                val text = mapper.getKeywordText(it)
                val dataMd5 = HashingUtils.getYabsMd5HalfHashUtf8(text)
                it.calculatedPhraseId = dataMd5
            }

        val mapping = mutableMapOf<BigInteger, BigInteger>()
        keywordBids.map { it.calculatedPhraseId }
            .distinct()
            .chunked(PHRASE_ID_SELECT_CHUNK)
            .map { historicalPhraseIdRepository.getPhraseIdByMd5(it) }
            .forEach(mapping::putAll)

        keywordBids
            .filter { mapping.containsKey(it.calculatedPhraseId) }
            .forEach { it.calculatedPhraseId = mapping[it.calculatedPhraseId] }
    }

    private fun saveChangedPhraseIds(shard: Int, bids: List<BsExportAbstractBid>) {
        val changes = bids.asSequence()
            .filterIsInstance<BsExportBidKeyword>()
            .filterNot { it.calculatedPhraseId.equals(it.phraseBsId) }
            .map { ChangedPhraseIdInfo(it.id, it.adGroupId, it.phraseBsId, it.calculatedPhraseId) }
            .distinctBy { it.id }
            .toList()

        if (changes.isEmpty()) {
            return
        }

        keywordRepository.updatePhraseId(shard, changes)
        bsAuctionStatRepository.updatePhraseId(shard, changes)
    }

    private data class SplitObjects(
        val updatedObjects: Map<BidObjectType, Set<BsExportBidsObject>>,
        val deletedObjects: List<BsExportBidsObject>,
    )

    /**
     * Разбивает объекты на группы "к изменению" и "к удалению":
     * * "объекты для удаления" - объекты с типом из [BidObjectType.BID_TABLE_TYPES],
     * которые не вставлялись/измененялись после удаления
     * * "объекты к изменению" - всё остальное
     */
    private fun splitObjects(logicObjects: List<BsExportBidsObject>): SplitObjects {
        val updatedObjectsByType = mutableMapOf<BidObjectType, MutableSet<BsExportBidsObject>>()
        val deletedObjectsByType = mutableMapOf<BidObjectType, MutableMap<Long, BsExportBidsObject>>()

        for (item in logicObjects) {
            val type = item.bidObjectType
            val updatedObjects = updatedObjectsByType.getOrPut(type, ::mutableSetOf)
            val deletedObjects = deletedObjectsByType.getOrPut(type, ::mutableMapOf)
            when {
                // удаление из таблиц ставок - объект помечается к удалению и убирается из списка к обновлению
                type.isDeletionSupported && item.isDeleted -> {
                    deletedObjects[item.id] = item
                    updatedObjects -= item
                }
                // вставка/изменение таблицы ставок - объект добавляется в список к обновлению и убирается из списка к удалению
                type.isDeletionSupported && !item.isDeleted -> {
                    deletedObjects -= item.id
                    updatedObjects += item
                }
                // вставка/изменение/удаление в других таблицах - объект всегда добавляется в список к обновлению
                else -> {
                    updatedObjects += item
                }
            }
        }

        return SplitObjects(updatedObjectsByType, deletedObjectsByType.values.flatMap { it.values })
    }

    fun fillMissedOrderId(shard: Int, bids: Collection<BsExportAbstractBid>) {
        val cids = bids.filter { it.orderId == null || it.orderId == 0L }
            .map { it.campaignId }
            .distinct()

        if (cids.isNotEmpty()) {
            val cid2orderid = bsOrderIdCalculator.calculateOrderIdIfNotExist(shard, cids)
            bids.filter { it.orderId == null || it.orderId == 0L }
                .forEach { it.orderId = cid2orderid[it.campaignId] }
        }
    }

    /*
     * для без cid мы пока не можем нормально обработать удаления
     * https://st.yandex-team.ru/DIRECT-132266#600f943e1d56eb0166385171
     * поэтому пытаемся вычислить cid по pid, а если не получается - остенется сирота
     * но это не очень страшно, потому что показов не будет
     */
    private fun convertToDeletes(
        shard: Int,
        mapper: BiddableShowConditionsYtRecordMapper,
        items: List<BsExportBidsObject>,
        now: BiddableShowConditionsYtRecordMapper.Now
    ): List<BiddableShowConditionWithCid> {
        val cids = mutableSetOf<Long>()
        val pids = mutableSetOf<Long>()
        for (item in items) {
            val cid = item.cid
            val pid = item.pid
            when {
                cid != null -> cids += cid
                pid != null -> pids += pid
                else -> logger.warn("Found object with both pid and cid not set: {}", item)
            }
        }

        val pid2cid = when {
            pids.isEmpty() -> mapOf<Long, Long>()
            else -> adGroupRepository.getCampaignIdsByAdGroupIds(shard, pids)
        }
        cids.addAll(pid2cid.values)

        val cid2oid = when {
            cids.isEmpty() -> mapOf<Long, Long>()
            else -> bsOrderIdCalculator.calculateOrderIdIfNotExist(shard, cids)
        }
        val ret = mutableListOf<BiddableShowConditionWithCid>()
        for (item in items) {
            val cid = when {
                item.cid != null -> item.cid
                else -> pid2cid[item.pid]
            }
            val orderId = when (cid) {
                null -> null
                in cid2oid -> cid2oid.getValue(cid)
                else -> bsOrderIdCalculator.calculateOrderId(cid)
            }
            if (orderId == null) {
                // без привязки к кампании фраза показываться не будет
                // со временем сделаем очистку таких сирот (аналогично BannerResources)
                logger.warn("Can't delete biddable show conditions - OrderID=null: {}", item)
            } else {
                ret += BiddableShowConditionWithCid(
                    mapper.createDeleteItem(orderId, item.pid!!, item.id, item.bidObjectType, now),
                    cid!!
                )
            }
        }
        return ret
    }

    private fun updateApplicableShowConditions(bidsRecords: List<BiddableShowConditionWithCid>) {
        val applicableRecords = bidsRecords.filter { it.bid.hasOrderId() && it.bid.orderId > 0L }
        logger.debug("{} records for send to YT", applicableRecords.size)
        if (applicableRecords.isNotEmpty()) {
            ytRepository.modify(applicableRecords.map { it.bid })
            logger.debug("records written to YT", applicableRecords.size)
            logBids(applicableRecords)
        }
    }

    private fun logBids(bids: List<BiddableShowConditionWithCid>) {
        if (logDisabledProp.getOrDefault(false)) {
            logger.warn("Skip logging due to property")
            return
        }
        val logsData = bids.map {
            LogBsExportEssData<BiddableShowCondition>()
                .withCid(it.cid)
                .withPid(it.bid.adGroupId)
                .withOrderId(it.bid.orderId)
                .withData(it.bid)
        }
        logBsExportEssService.logData(logsData, LOG_TYPE)
    }

    companion object {
        private val logger = LoggerFactory.getLogger(BsExportBiddableShowConditionsService::class.java)
        private const val LOG_TYPE = "biddable_show_conditions"
        private const val PHRASE_ID_SELECT_CHUNK = 1000
    }
}
