package ru.yandex.direct.logicprocessor.processors.bsexport.campaign

import java.time.Clock
import java.util.IdentityHashMap
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.adv.direct.campaign.Campaign
import ru.yandex.direct.bstransport.yt.repository.campaign.CampaignDeleteYtRepository
import ru.yandex.direct.bstransport.yt.repository.campaign.CampaignYtRepository
import ru.yandex.direct.bstransport.yt.utils.CaesarIterIdGenerator
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData
import ru.yandex.direct.common.log.service.LogBsExportEssService
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.BsExportCampaignObject
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.CampaignResourceType
import ru.yandex.direct.logicprocessor.processors.bsexport.campaign.container.CampaignWithBuilder
import ru.yandex.direct.logicprocessor.processors.bsexport.campaign.handler.CampaignDeleteHandler
import ru.yandex.direct.logicprocessor.processors.bsexport.campaign.handler.ICampaignResourceHandler
import ru.yandex.direct.logicprocessor.processors.bsexport.campaign.utils.BsExportCampaignOrderIdLoader
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.SupportedCampaignsService

@Service
class BsExportCampaignService(
    private val logBsExportEssService: LogBsExportEssService,
    private val caesarIterIdGenerator: CaesarIterIdGenerator,
    private val ytRepository: CampaignYtRepository,
    private val ytDeleteRepository: CampaignDeleteYtRepository,
    private val clock: Clock = Clock.systemUTC(),
    private val resourceHandlers: List<ICampaignResourceHandler<*>>,
    private val deleteHandler: CampaignDeleteHandler,
    private val campaignOrderIdLoader: BsExportCampaignOrderIdLoader,
    private val supportedCampaignsRepository: SupportedCampaignsService,
) {

    private val resourceHandlersMap: Map<CampaignResourceType, List<ICampaignResourceHandler<*>>> = resourceHandlers
        .associateBy({ it.resourceType }) { listOf(it) }

    fun processCampaigns(shard: Int, logicObjects: List<BsExportCampaignObject>) {
        val (deletedObjects, updatedObjects) = logicObjects
            .partition {
                it.campaignResourceType == CampaignResourceType.CAMPAIGN_DELETE
            }
        val iterId = caesarIterIdGenerator.generateCaesarIterId()
        if (updatedObjects.isNotEmpty()) {
            modifyCampaigns(shard, updatedObjects, iterId)
        }
        if (deletedObjects.isNotEmpty()) {
            markCampaignsAsDeleted(shard, deletedObjects, iterId)
        }
    }

    private fun modifyCampaigns(shard: Int, updatedObjects: Collection<BsExportCampaignObject>, iterId: Long) {
        val campaignsWithBuilders = loadCampaignsFromDb(shard, updatedObjects, iterId)

        try {
            modifyLoadedCampaigns(shard, campaignsWithBuilders)
        } catch (e: Exception) {
            logger.error("Campaigns processing failed for campaign ids ${campaignsWithBuilders.keys}")
            throw e
        }
    }

    private fun loadCampaignsFromDb(
        shard: Int,
        updatedObjects: Collection<BsExportCampaignObject>,
        iterId: Long,
    ): Map<Long, CampaignWithBuilder> {
        // объекты группируются по хэндлеру, для каждого хэндлера
        // ищутся ID кампаний, которые надо загрузить из базы
        val campaignIdsToLoad: Set<Long> = updatedObjects
            .flatMap { obj -> getHandlers(obj.campaignResourceType).map { handler -> handler to obj } }
            .groupByTo(IdentityHashMap(), { it.first }) { it.second }
            .flatMapTo(hashSetOf()) { (handler, objects) -> handler.getCampaignsIdsToLoad(shard, objects).toSet() }
        val dbCampaigns = supportedCampaignsRepository.getCampaignsByIdTyped(shard, campaignIdsToLoad)

        // если кампании нет в базе - её не будет тут
        val orderIdByCids = campaignOrderIdLoader.getOrderIdForExistingCampaigns(
            shard,
            dbCampaigns.keys,
            knownCampaignsObjects = updatedObjects,
        )

        val updateTime = clock.instant().epochSecond
        return orderIdByCids.entries
            .filter { (cid, _) -> cid in dbCampaigns }
            .associateBy({ it.key }) { (cid, orderId) ->
                val campaign = checkNotNull(dbCampaigns[cid]) {
                    "Campaign must be known there"
                }
                val builder = Campaign.newBuilder()
                    .setExportId(cid)
                    .setOrderId(orderId)
                    .setIterId(iterId)
                    .setUpdateTime(updateTime)
                CampaignWithBuilder(campaign, builder)
            }
    }

    private fun modifyLoadedCampaigns(shard: Int, campaignsWithBuilders: Map<Long, CampaignWithBuilder>) {
        if (campaignsWithBuilders.isEmpty()) {
            return
        }

        for (handler in resourceHandlers) {
            handler.handle(shard, campaignsWithBuilders)
        }

        val campaigns = campaignsWithBuilders.map { (_, v) -> v.builder.build() }
        ytRepository.modify(campaigns)
        logCampaigns(campaigns)
    }

    private fun markCampaignsAsDeleted(shard: Int, deletedObjects: Collection<BsExportCampaignObject>, iterId: Long) {
        val deleteTime = clock.instant().epochSecond

        val deletedCampaignIds = deleteHandler.getDeletedCampaignIds(shard, deletedObjects)
        val orderIdsByCids = campaignOrderIdLoader.calculateOrderIdForCampaigns(
            deletedCampaignIds,
            deletedObjects,
        )

        val campaigns = orderIdsByCids
            .map { (cid, orderId) ->
                Campaign.newBuilder()
                    .setExportId(cid)
                    .setOrderId(orderId)
                    .setIterId(iterId)
                    .setUpdateTime(deleteTime)
                    .setDeleteTime(deleteTime)
                    .build()
            }

        if (campaigns.isNotEmpty()) {
            ytDeleteRepository.modify(campaigns)
            logCampaigns(campaigns)
        }
    }

    private fun logCampaigns(campaigns: List<Campaign>) {
        val recordsToLog = campaigns
            .map {
                LogBsExportEssData<Campaign>()
                    .withCid(it.exportId)
                    .withOrderId(it.orderId)
                    .withData(it)
            }

        logBsExportEssService.logData(recordsToLog, DATA_TYPE)
    }

    private fun getHandlers(resourceType: CampaignResourceType?) =
        when (resourceType) {
            CampaignResourceType.ALL -> resourceHandlers
            null -> listOf()
            else -> resourceHandlersMap[resourceType] ?: listOf()
        }

    companion object {
        private val logger = LoggerFactory.getLogger(BsExportCampaignService::class.java)
        private const val DATA_TYPE = "campaign"
    }
}
