package ru.yandex.direct.logicprocessor.processors.bsexport.campaign.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.campaign.Campaign
import ru.yandex.adv.direct.campaign.CampaignMetatype
import ru.yandex.adv.direct.campaign.CampaignSource
import ru.yandex.direct.core.entity.campaign.model.CampaignMetatype as CoreCampaignMetatype
import ru.yandex.direct.core.entity.campaign.model.CampaignSource as CoreCampaignSource
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.CampaignResourceType
import ru.yandex.direct.logicprocessor.processors.bsexport.campaign.container.CampaignCommonFields
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.CampaignNameTransliterator

@Component
open class CampaignCommonFieldsHandler(
    private val transliterator: CampaignNameTransliterator,
) : TypedCampaignResourceHandler<CampaignCommonFields, CommonCampaign>() {

    override val resourceType = CampaignResourceType.COMMON_FIELDS

    override val campaignToLoad = CommonCampaign::class.java

    override fun fillExportObject(resource: CampaignCommonFields, builder: Campaign.Builder) {
        builder.apply {
            clientId = resource.clientId
            archive = resource.isArchive
            stop = resource.isStop
            type = resource.type
            name = resource.name
            latName = resource.latName
            source = resource.source.number
            metatype = resource.metatype.number
        }
    }

    override fun mapCampaignToResource(campaign: CommonCampaign): CampaignCommonFields {
        val cleanedCampName = cleanCampaignName(campaign.name)
        val campaignCommonFieldsBuilder = CampaignCommonFields.builder()
            .withClientId(campaign.clientId)
            .withArchive(campaign.statusArchived)
            .withStop(!campaign.statusShow)
            .withName(cleanedCampName)
            .withLatName(transliterator.translit(cleanedCampName)
                .replace("[^a-zA-Z0-9_]".toRegex(), "_"))
            .withType(CampaignType.toSource(campaign.type)!!.literal)
            .withSource(CampaignSourceConverter.toExportSource(campaign.source))
            .withMetatype(CampaignMetatypeConverter.toExportMetatype(campaign.metatype))
        return campaignCommonFieldsBuilder.build()
    }

    private fun cleanCampaignName(name: String): String =
        name.replace(CampaignConstants.DISALLOW_CAMPAIGN_NAME_LETTERS_RE.toRegex(), "")
}

object CampaignSourceConverter {
    private val coreToExportSources: Map<CoreCampaignSource, CampaignSource> = mapOf(
        CoreCampaignSource.DIRECT to CampaignSource.CAMPAIGN_SOURCE_DIRECT,
        CoreCampaignSource.UAC to CampaignSource.CAMPAIGN_SOURCE_UAC,
        CoreCampaignSource.USLUGI to CampaignSource.CAMPAIGN_SOURCE_USLUGI,
        CoreCampaignSource.EDA to CampaignSource.CAMPAIGN_SOURCE_EDA,
        CoreCampaignSource.GEO to CampaignSource.CAMPAIGN_SOURCE_GEO,
        CoreCampaignSource.DC to CampaignSource.CAMPAIGN_SOURCE_DC,
        CoreCampaignSource.API to CampaignSource.CAMPAIGN_SOURCE_API,
        CoreCampaignSource.XLS to CampaignSource.CAMPAIGN_SOURCE_XLS,
        CoreCampaignSource.ZEN to CampaignSource.CAMPAIGN_SOURCE_ZEN,
        CoreCampaignSource.WIDGET to CampaignSource.CAMPAIGN_SOURCE_WIDGET,
    )

    fun toExportSource(source: CoreCampaignSource): CampaignSource {
        return coreToExportSources[source]
            ?: throw IllegalArgumentException("Unknown campaign source: $source")
    }
}

object CampaignMetatypeConverter {
    private val coreToExportMetatypes: Map<CoreCampaignMetatype, CampaignMetatype> = mapOf(
        CoreCampaignMetatype.DEFAULT_ to CampaignMetatype.CAMPAIGN_METATYPE_DEFAULT,
        CoreCampaignMetatype.ECOM to CampaignMetatype.CAMPAIGN_METATYPE_ECOM,
    )

    fun toExportMetatype(metatype: CoreCampaignMetatype): CampaignMetatype {
        return coreToExportMetatypes[metatype]
            ?: throw IllegalArgumentException("Unknown campaign metatype: $metatype")
    }
}
