package ru.yandex.direct.logicprocessor.processors.bsexport.campaign.handler

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.CampaignResourceType
import ru.yandex.direct.logicprocessor.processors.bsexport.campaign.container.CampaignWithBuilder

/**
 * Выгружается признак наличия мобильных целей в стратегии или ключевых целях.
 *
 * Учитываются только "новые" мобильные цели, которые создаются для мобильного приложения,
 * фиксированные мобильные цели (3, 4 и т.д.) не учитываются.
 *
 * Пока важен только признак наличия мобильных целей. Вне Директа это учитывается как наличие любого
 * значения в списке идентификаторов мобильных приложений
 *
 * Сам список сейчас кажется ещё и неправильного типа и ID мобильного приложения может не влезть в Int,
 * если будем переделывать на экспорт ID, то нужно будет переделывать тип поля в proto.
 */

private typealias MobileAppsMark = List<Int>?

val HasMobileGoalsValue = listOf(1) // просто отметка, что есть мобильные цели

@Component
class CampaignMobileMarkHandler : ICampaignResourceHandler<MobileAppsMark> {

    override val resourceType: CampaignResourceType = CampaignResourceType.MOBILE_GOALS

    override fun handle(shard: Int, campaigns: Map<Long, CampaignWithBuilder>) {
        for ((campaign, builder) in campaigns.values) {
            val goalIds = mutableSetOf<Long>()
            if (campaign is CampaignWithMeaningfulGoals) {
                goalIds.addAll(extractMeaningfulGoalIds(campaign))
            }
            if (campaign is CampaignWithStrategy) {
                val goalId = extractStrategyGoalId(campaign)
                if (goalId != null) {
                    goalIds.add(goalId)
                }
            }
            if (goalIds.map { Goal.computeType(it) }.any { it == GoalType.MOBILE }) {
                builder.mobileAppIdsBuilder.addAllValues(HasMobileGoalsValue)
            }
        }
    }

    private fun extractMeaningfulGoalIds(campaign: CampaignWithMeaningfulGoals): List<Long> {
        return campaign.meaningfulGoals?.map { it.goalId } ?: listOf()
    }

    private fun extractStrategyGoalId(campaign: CampaignWithStrategy): Long? {
        return campaign.strategy?.strategyData?.goalId
    }
}
