package ru.yandex.direct.logicprocessor.processors.bsexport.campaign.handler

import com.google.common.base.Suppliers
import com.google.common.net.InetAddresses
import com.google.protobuf.InvalidProtocolBufferException
import com.google.protobuf.util.JsonFormat
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.adv.direct.expression2.TargetingExpressionAtom
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.net.IpUtils.ipFromString
import ru.yandex.direct.core.entity.additionaltargetings.model.CampAdditionalTargeting
import ru.yandex.direct.core.entity.additionaltargetings.model.ClientAdditionalTargeting
import ru.yandex.direct.core.entity.additionaltargetings.repository.CampAdditionalTargetingsRepository
import ru.yandex.direct.core.entity.additionaltargetings.repository.ClientAdditionalTargetingsRepository
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAllowedPageIds
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandSafety
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDayBudget
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDisabledDomainsAndSsp
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDisabledVideoPlacements
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDisallowedPageIds
import ru.yandex.direct.core.entity.campaign.model.CampaignWithFrontpageTypes
import ru.yandex.direct.core.entity.campaign.model.CampaignWithImpressionRate
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.InternalCampaign
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType
import ru.yandex.direct.core.entity.pricepackage.model.ViewType
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.BsExportCampaignObject
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.CampaignResourceType
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.ClientAdditionalTargetingsInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.campaign.container.CampaignWithBuilder
import ru.yandex.direct.logicprocessor.processors.bsexport.campaign.utils.DisallowedTargetTypesCalculator
import ru.yandex.direct.logicprocessor.processors.bsexport.common.ShowConditionUtils.calcContextID
import ru.yandex.direct.logicprocessor.processors.bsexport.common.ShowConditionUtils.targetingExpressionSort
import ru.yandex.direct.logicprocessor.processors.bsexport.common.TimeTargetBsExportUtils.addTargetTimeCondition
import ru.yandex.direct.utils.CommonUtils.nvl
import java.math.BigDecimal
import java.net.IDN
import java.time.Duration
import java.time.LocalDate
import java.time.ZoneId
import java.util.concurrent.TimeUnit

val MSK_ZONE: ZoneId = ZoneId.of("Europe/Moscow")
const val OTHER_ADTYPES_MASK = ((1 shl 1) + (1 shl 2)).toString()
const val VIDEO_ADTYPES_MASK = ((1 shl 3) + (1 shl 4) + (1 shl 5) + (1 shl 7)).toString()

val PAGE_TYPE_TO_PAGE_IDS = mapOf(
    FrontpageCampaignShowType.FRONTPAGE to listOf(345620, 674114, 1638690, 1638708, 1654537),
    FrontpageCampaignShowType.FRONTPAGE_MOBILE to listOf(349254, 674124, 1638693, 1638711, 1654534, 1654540),
    FrontpageCampaignShowType.BROWSER_NEW_TAB to listOf(526194, 526196, 674134, 674137),
)
val TYPE_VIEW_TO_PAGE_IDS = mapOf(
    ViewType.DESKTOP to PAGE_TYPE_TO_PAGE_IDS[FrontpageCampaignShowType.FRONTPAGE],
    ViewType.MOBILE to PAGE_TYPE_TO_PAGE_IDS[FrontpageCampaignShowType.FRONTPAGE_MOBILE],
    ViewType.NEW_TAB to PAGE_TYPE_TO_PAGE_IDS[FrontpageCampaignShowType.BROWSER_NEW_TAB]
)

@Component
class CampaignShowConditionsHandler(
    private val disallowedTargetTypesCalculator: DisallowedTargetTypesCalculator,
    private val sspPlatformsRepository: SspPlatformsRepository,
    private val cryptaSegmentRepository: CryptaSegmentRepository,
    private val retargetingConditionRepository: RetargetingConditionRepository,
    private val campAdditionalTargetingsRepository: CampAdditionalTargetingsRepository,
    private val clientAdditionalTargetingsRepository: ClientAdditionalTargetingsRepository,
    private val campaignRepository: CampaignRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) : ICampaignResourceHandler<TargetingExpression> {

    private val logger = LoggerFactory.getLogger(CampaignShowConditionsHandler::class.java)
    private val sspMap = Suppliers.memoizeWithExpiration(
        sspPlatformsRepository::getSspTitlesToIds,
        15, TimeUnit.MINUTES
    )
    private val cryptaGoals = Suppliers.memoizeWithExpiration(
        cryptaSegmentRepository::getBrandSafety,
        15, TimeUnit.MINUTES
    )
    private val splitUriRegex = Regex("""^([a-zA-Z\s]+://)?((?:[\pL\pN][\pL\pN\-_]*\.)+[\pL\pN\-]+\.?)(.*)$""")

    private val oldRfDisabledPlaceIdsProp = ppcPropertiesSupport.get(
        PpcPropertyNames.BS_EXPORT_OLD_RF_DISABLED_PLACE_IDS,
        Duration.ofMinutes(5),
    )

    override val resourceType = CampaignResourceType.SHOW_CONDITIONS

    override fun getCampaignsIdsToLoad(shard: Int, objects: Collection<BsExportCampaignObject>): Collection<Long> {
        val brandSafetyIds = objects.mapNotNull { it.brandsafetyRetCondId }
        val brandSafetyCampaignIds = retargetingConditionRepository.getCampaignIdsByBrandSafetyRetConditions(
            shard, brandSafetyIds
        )

        val clientIds = objects
            .mapNotNull {
                (it.additionalInfo as? ClientAdditionalTargetingsInfo)?.clientId
            }.map(ClientId::fromLong)
        val clientCampaignIds = campaignRepository.getCampaignIdsByClientIds(shard, clientIds)

        return brandSafetyCampaignIds + clientCampaignIds + objects.mapNotNull(BsExportCampaignObject::getCid)
    }

    private inline fun <reified T : BaseCampaign> addConditionWithCast(
        crossinline addCondition: ((T, TargetingExpression.Builder) -> Unit)
    )
        : ((BaseCampaign, TargetingExpression.Builder) -> Unit) {
        return { baseCampaign, builder ->
            if (baseCampaign is T) {
                addCondition(baseCampaign, builder)
            }
        }
    }

    val conditionHandlers = listOf(
        addConditionWithCast(this::addDisabledIpsCondition),
        addConditionWithCast(this::addTargetTypeCondition),
        addConditionWithCast(this::addStopTimeCondition),
        addConditionWithCast(this::addDomainsCondition),
        addConditionWithCast(this::addVideoDomainsCondition),
        addConditionWithCast(this::addTargetTimeCondition),
        this::addPageIdCondition,
        addConditionWithCast(this::addDisabledSspCondition),
        addConditionWithCast(this::addIOsCondition),
        addConditionWithCast(this::addBrandSafetyCondition),
        addConditionWithCast(this::addDisabledPageIdCondition),
        addConditionWithCast(this::addRfCondition),
        addConditionWithCast(this::addInternalPageIdsCondition),
    )

    override fun handle(shard: Int, campaigns: Map<Long, CampaignWithBuilder>) {
        val mapCampAddTargetings = campAdditionalTargetingsRepository
            .findByCids(shard, campaigns.keys)
            .groupBy { it.cid }

        val clientIds = campaigns
            .map { it.value.campaign }
            .filterIsInstance<CommonCampaign>()
            .mapNotNull { it.clientId }
            .distinct()
        val mapClientAddTargetings = clientAdditionalTargetingsRepository
            .findByClientIds(shard, clientIds)
            .groupBy { it.clientId }

        for (campaignWithBuilder in campaigns.values) {
            val condBuilder = TargetingExpression.newBuilder()
            conditionHandlers.forEach {
                it(campaignWithBuilder.campaign, condBuilder)
            }
            addCampaignAdditionalTargetingCondition(campaignWithBuilder.campaign, condBuilder, mapCampAddTargetings)
            if (campaignWithBuilder.campaign is CommonCampaign)
                addClientAdditionalTargetingCondition(campaignWithBuilder.campaign, condBuilder, mapClientAddTargetings)
            campaignWithBuilder.builder.showConditions = targetingExpressionSort(condBuilder.build())
            campaignWithBuilder.builder.contextId = calcContextID(campaignWithBuilder.builder.showConditions)
        }
    }

    private fun toAtom(keyword: KeywordEnum, operation: OperationEnum, value: String): TargetingExpressionAtom.Builder {
        return TargetingExpressionAtom.newBuilder()
            .setKeyword(keyword.number)
            .setOperation(operation.number)
            .setValue(value)
    }

    private fun addDisabledIpsCondition(campaign: CommonCampaign, builder: TargetingExpression.Builder) {
        if (campaign.disabledIps != null) {
            for (ipStr in campaign.disabledIps) {
                var intIp: Int
                try {
                    intIp = InetAddresses.coerceToInteger(ipFromString(ipStr))
                } catch (e: IllegalArgumentException) {
                    logger.error("Invalid ip address: $ipStr")
                    continue
                }

                builder.addAnd(
                    TargetingExpression.Disjunction.newBuilder().addOr(
                        toAtom(KeywordEnum.ClientIp, OperationEnum.NotEqual, Integer.toUnsignedString(intIp))
                    )
                )
            }
        }
    }

    private fun addTargetTypeCondition(campaign: CommonCampaign, builder: TargetingExpression.Builder) {
        val disallowedTypes = disallowedTargetTypesCalculator.calculate(campaign)
        disallowedTypes.map {
            builder.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    toAtom(KeywordEnum.PageTargetType, OperationEnum.NotEqual, it.toString())
                )
            )
        }
    }

    private fun toEndOfMskDay(date: LocalDate): Long {
        return date.atTime(23, 59, 59).atZone(MSK_ZONE).toEpochSecond()
    }

    private fun addStopTimeCondition(campaign: CommonCampaign, builder: TargetingExpression.Builder) {
        var stopTime: Long? = null
        if (campaign.endDate != null) {
            stopTime = toEndOfMskDay(campaign.endDate)
        }
        if (campaign is CampaignWithStrategy
            && campaign.strategy != null
            && campaign.strategy.strategyData != null
            && campaign.strategy.strategyData.name != null
        ) {
            if (stopTime != null && campaign.strategy.strategyData.name == CampaignsStrategyName.period_fix_bid.literal
                && campaign.strategy.strategyData.autoProlongation ?: 0 != 0L
            ) {
                stopTime = toEndOfMskDay(campaign.endDate.plusDays(2))
            }

            if (campaign.strategy.strategyData.finish != null
                && campaign.strategy.strategyData.autoProlongation ?: 0 == 0L
                && (
                    (campaign.strategy.autobudget == CampaignsAutobudget.YES
                        && campaign.strategy.strategyData.name in listOf(
                        CampaignsStrategyName.autobudget_avg_cpv_custom_period.literal,
                        CampaignsStrategyName.autobudget_max_impressions_custom_period.literal,
                        CampaignsStrategyName.autobudget_max_reach_custom_period.literal
                    ))
                        || (!(campaign is CampaignWithDayBudget && campaign.dayBudget != null && campaign.dayBudget > BigDecimal.ZERO)
                        && campaign.strategy.strategyData.name == CampaignsStrategyName.period_fix_bid.literal)
                    )
            ) {
                val abStopTime = toEndOfMskDay(campaign.strategy.strategyData.finish)
                if (stopTime == null || abStopTime < stopTime) {
                    stopTime = abStopTime
                }
            }
        }
        if (stopTime == null) {
            return
        }
        builder.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                toAtom(KeywordEnum.Unixtime, OperationEnum.Less, stopTime.toString())
            )
        )
    }

    /**
     * Основано на перловом Yandex::IDN::idn_to_ascii
     * @see <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/infra/direct-utils/yandex-lib/idn/lib/Yandex/IDN.pm?rev=r8715111#L18">
     *     Yandex::IDN::idn_to_ascii</a>
     */
    fun domainToASCII(domain: String): String? {
        try {
            val parts = splitUriRegex.find(domain.trim())?.groupValues ?: throw IllegalArgumentException()
            return parts[1] + IDN.toASCII(parts[2]) + parts[3]
        } catch (e: IllegalArgumentException) {
            logger.warn("Invalid disabled domain \"$domain\"")
        }
        return null
    }

    private fun addDomainsCondition(campaign: CampaignWithDisabledDomainsAndSsp, builder: TargetingExpression.Builder) {
        if (campaign.disabledDomains != null) {
            val domains = campaign.disabledDomains.mapNotNull(this::domainToASCII).distinct()
            domains.map {
                val disjBuilder = TargetingExpression.Disjunction.newBuilder().addOr(
                    toAtom(KeywordEnum.Domain, OperationEnum.DomainNotLike, it)
                )
                if (campaign.type in CampaignTypeKinds.CPM) {
                    disjBuilder.addOr(
                        toAtom(KeywordEnum.AdType, OperationEnum.NegationOfBitwiseAnd, OTHER_ADTYPES_MASK)
                    )
                }
                builder.addAnd(disjBuilder)
            }
        }
    }

    private fun addVideoDomainsCondition(
        campaign: CampaignWithDisabledVideoPlacements,
        builder: TargetingExpression.Builder
    ) {
        if (campaign.disabledVideoPlacements != null) {
            val domains = campaign.disabledVideoPlacements.mapNotNull(this::domainToASCII).distinct()
            domains.map {
                builder.addAnd(
                    TargetingExpression.Disjunction.newBuilder().addOr(
                        toAtom(KeywordEnum.Domain, OperationEnum.DomainNotLike, it)
                    ).addOr(
                        toAtom(KeywordEnum.AdType, OperationEnum.NegationOfBitwiseAnd, VIDEO_ADTYPES_MASK)
                    )
                )
            }
        }
    }

    private fun hourToLetter(hour: Int): String {
        return ('A' + hour).toString()
    }

    // original: https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/TimeTarget.pm?rev=r8318553#L739
    private fun addTargetTimeCondition(campaign: CommonCampaign, builder: TargetingExpression.Builder) {
        val timeTarget = campaign.timeTarget ?: return
        addTargetTimeCondition(timeTarget, builder)
    }

    private fun addPageIdCondition(campaign: BaseCampaign, builder: TargetingExpression.Builder) {
        val builderDisj = TargetingExpression.Disjunction.newBuilder()
        var empty = true
        if (campaign is CampaignWithAllowedPageIds && !campaign.allowedPageIds.isNullOrEmpty()) {
            empty = false
            builderDisj.addAllOr(
                campaign.allowedPageIds.map { toAtom(KeywordEnum.PageId, OperationEnum.Equal, it.toString()).build() }
            )
        }
        if (campaign is CampaignWithFrontpageTypes && !campaign.allowedFrontpageType.isNullOrEmpty()) {
            empty = false
            builderDisj.addAllOr(
                campaign.allowedFrontpageType.flatMap {
                    PAGE_TYPE_TO_PAGE_IDS[it]!!.map { pageId ->
                        toAtom(KeywordEnum.PageId, OperationEnum.Equal, pageId.toString()).build()
                    }
                }
            )
        }
        if (campaign is CampaignWithPricePackage && !nvl(
                campaign.flightTargetingsSnapshot?.viewTypes?.isNullOrEmpty(),
                true
            )
        ) {
            empty = false
            builderDisj.addAllOr(
                campaign.flightTargetingsSnapshot.viewTypes.flatMap {
                    TYPE_VIEW_TO_PAGE_IDS[it]!!.map { pageId ->
                        toAtom(KeywordEnum.PageId, OperationEnum.Equal, pageId.toString()).build()
                    }
                }
            )
        }
        if (!empty) {
            builder.addAnd(builderDisj)
        }
    }

    private fun addDisabledSspCondition(
        campaign: CampaignWithDisabledDomainsAndSsp,
        builder: TargetingExpression.Builder
    ) {
        if (campaign.disabledSsp != null) {
            campaign.disabledSsp.forEach {
                val sspId = sspMap.get()[it.trim()]
                if (sspId != null) {
                    builder.addAnd(
                        TargetingExpression.Disjunction.newBuilder().addOr(
                            toAtom(KeywordEnum.SspId, OperationEnum.NotEqual, sspId.toString())
                        )
                    )
                } else {
                    logger.error("Unknown SSP platform: $it")
                }
            }
        }
    }

    private fun addIOsCondition(campaign: MobileContentCampaign, builder: TargetingExpression.Builder) {
        if (campaign.strategy?.strategyName == StrategyName.AUTOBUDGET_AVG_CPI
            || (campaign.strategy?.strategyName == StrategyName.AUTOBUDGET
                && campaign.strategy?.strategyData?.goalId != null)
        ) {
            builder.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    toAtom(KeywordEnum.OsFamilyAndVersion, OperationEnum.NotMatchNameAndVersionRange, "3:14005:")
                )
            )
        } else if (campaign.isSkadNetworkEnabled == true) {
            builder.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    toAtom(KeywordEnum.OsFamilyAndVersion, OperationEnum.MatchNameAndVersionRange, "3:14000:")
                )
            )
        } else if (campaign.isNewIosVersionEnabled != true) {
            builder.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    toAtom(KeywordEnum.OsFamilyAndVersion, OperationEnum.NotMatchNameAndVersionRange, "3:14005:")
                )
            )
        }
    }

    private fun addBrandSafetyCondition(campaign: CampaignWithBrandSafety, builder: TargetingExpression.Builder) {
        if (campaign.brandSafetyCategories.isNullOrEmpty()) {
            return
        }

        for (goalId in campaign.brandSafetyCategories) {
            val cryptaGoal = cryptaGoals.get()[goalId]
                ?: throw RuntimeException("Non-brandsafety goal in brandsafety retargeting condition, campaignId = ${campaign.id}")

            builder.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    toAtom(KeywordEnum.BrandSafetyCategories, OperationEnum.NotEqual, cryptaGoal.keywordValue)
                )
            )
        }
    }

    private fun addDisabledPageIdCondition(
        campaign: CampaignWithDisallowedPageIds,
        builder: TargetingExpression.Builder
    ) {
        campaign.disallowedPageIds?.forEach { pageId ->
            builder.addAnd(
                TargetingExpression.Disjunction.newBuilder().addOr(
                    toAtom(KeywordEnum.PageId, OperationEnum.NotEqual, pageId.toString())
                )
            )
        }
    }

    private fun addRfCondition(campaign: CampaignWithImpressionRate, builder: TargetingExpression.Builder) {
        if (campaign !is InternalCampaign || campaign.impressionRateCount ?: 0 == 0) {
            return
        }
        if (campaign is CommonCampaign && !needExportCampaignRfCondition(campaign.placeId)) {
            return
        }

        val rfReset = campaign.impressionRateIntervalDays ?: 0
        builder.addAnd(
            TargetingExpression.Disjunction.newBuilder().addOr(
                TargetingExpressionAtom.newBuilder()
                    .setKeyword(KeywordEnum.FrequencyDay.number)
                    .setOperation(OperationEnum.Less.number)
                    .setValue(campaign.impressionRateCount.toString())
                    .build()
            ).addOr(
                TargetingExpressionAtom.newBuilder()
                    .setKeyword(KeywordEnum.FreqExpire.number)
                    .setOperation(OperationEnum.GreaterOrEqual.number)
                    .setValue(rfReset.toString())
                    .build()
            )
        )
    }

    /**
     * Определяет нужно ли экспортировать старый РФ, вызывается только для кампаний внутренней рекламы
     * false - если включена фильтрация и placeId кампании находится в черном списке
     * true - иначе
     */
    private fun needExportCampaignRfCondition(placeId: Long?): Boolean {
        return !oldRfDisabledPlaceIdsProp.getOrDefault(emptySet()).contains(placeId)
    }

    private fun addInternalPageIdsCondition(campaign: InternalCampaign, builder: TargetingExpression.Builder) {
        if (campaign.pageId.isNullOrEmpty()) {
            return
        }

        builder.addAnd(TargetingExpression.Disjunction.newBuilder().addAllOr(
            campaign.pageId.filter { it > 0 }.map { pageId ->
                toAtom(KeywordEnum.PageId, OperationEnum.Equal, pageId.toString()).build()
            }
        ))
    }

    private fun addCampaignAdditionalTargetingCondition(
        campaign: BaseCampaign,
        builder: TargetingExpression.Builder,
        mapCampAddTargetings: Map<Long, List<CampAdditionalTargeting>>
    ) {
        mapCampAddTargetings[campaign.id]?.forEach {
            try {
                val partBuilder = TargetingExpression.newBuilder()
                JsonFormat.parser().merge(it.data, partBuilder)
                builder.addAllAnd(partBuilder.build().andList)
            } catch (e: InvalidProtocolBufferException) {
                logger.error("Failed to parse campaign additional targeting ${it.id}")
            }
        }
    }

    private fun addClientAdditionalTargetingCondition(
        campaign: CommonCampaign,
        builder: TargetingExpression.Builder,
        mapClientAddTargetings: Map<Long, List<ClientAdditionalTargeting>>
    ) {
        mapClientAddTargetings[campaign.clientId]?.forEach {
            try {
                val partBuilder = TargetingExpression.newBuilder()
                JsonFormat.parser().merge(it.data, partBuilder)
                builder.addAllAnd(partBuilder.build().andList)
            } catch (e: InvalidProtocolBufferException) {
                logger.error("Failed to parse client additional targeting ${it.id}")
            }
        }
    }
}
