package ru.yandex.direct.logicprocessor.processors.bsexport.campaign.utils

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.BsExportCampaignObject

@Service
class BsExportCampaignOrderIdLoader(
    private val bsOrderIdCalculator: BsOrderIdCalculator,
) {
    /**
     * Возвращает OrderID кампаний из [requestedCampaigns], которые есть в MySQL
     *
     * Если у кампании был проставлен [BsExportCampaignObject.orderId], то просто возвращается известное значение
     * Иначе OrderID вычисляется
     */
    fun getOrderIdForExistingCampaigns(
        shard: Int,
        requestedCampaigns: Collection<Long>,
        knownCampaignsObjects: Collection<BsExportCampaignObject>,
    ): Map<Long, Long> {
        val knownOrderId = extractOrderId(knownCampaignsObjects)
        val unknownOrderId = bsOrderIdCalculator.calculateOrderIdIfNotExist(
            shard,
            requestedCampaigns - knownOrderId.keys,
        )
        return knownOrderId + unknownOrderId
    }

    /**
     * Вычисляет OrderID кампаний из [requestedCampaigns]
     *
     * Если у кампании был проставлен [BsExportCampaignObject.orderId], то просто возвращается известное значение
     * Иначе OrderID вычисляется
     */
    fun calculateOrderIdForCampaigns(
        requestedCampaigns: Collection<Long>,
        knownCampaignsObjects: Collection<BsExportCampaignObject>,
    ): Map<Long, Long> {
        val knownOrderId = extractOrderId(knownCampaignsObjects)
        return requestedCampaigns
            .associateWith { cid ->
                knownOrderId[cid] ?: bsOrderIdCalculator.calculateOrderId(cid)
            }
    }

    private fun extractOrderId(objects: Collection<BsExportCampaignObject>): Map<Long, Long> =
        objects
            .asSequence()
            .filter { it.orderId != null && it.orderId != 0L }
            .associateBy(
                keySelector = BsExportCampaignObject::getCid,
                valueTransform = BsExportCampaignObject::getOrderId
            )
}
