package ru.yandex.direct.logicprocessor.processors.bsexport.common

import ru.yandex.adv.direct.expression.keywords.KeywordEnum
import ru.yandex.adv.direct.expression.operations.OperationEnum
import ru.yandex.adv.direct.expression2.TargetingExpression
import ru.yandex.adv.direct.expression2.TargetingExpressionAtom
import ru.yandex.direct.utils.HashingUtils.getMd5HalfMix
import java.math.BigInteger
import java.nio.ByteBuffer
import java.nio.ByteOrder
import java.security.MessageDigest

object ShowConditionUtils {

    private val atomComparator =
        Comparator { a: TargetingExpressionAtom, b: TargetingExpressionAtom ->
            when {
                a.keyword != b.keyword -> a.keyword.compareTo(b.keyword)
                a.operation != b.operation -> a.operation.compareTo(b.operation)
                else -> a.value.compareTo(b.value)
            }
        }

    private val disjunctionComparator =
        Comparator { a: TargetingExpression.Disjunction, b: TargetingExpression.Disjunction ->
            if (a.orCount != b.orCount)
                return@Comparator a.orCount.compareTo(b.orCount)
            for (i in 0 until a.orCount) {
                val eq = atomComparator.compare(a.orList[i], b.orList[i])
                if (eq != 0)
                    return@Comparator eq
            }
            0
        }

    /**
     * Сортировка атомов внутри orList и дизъюнктов внутри andList по алгоритму БК
     * https://a.yandex-team.ru/arc/trunk/arcadia/yabs/server/cs/libs/import_operations/ad_groups_show_conditions.cpp#L7
     */
    fun targetingExpressionSort(proto: TargetingExpression): TargetingExpression =
        TargetingExpression.newBuilder()
            .addAllAnd(
                proto.andList
                    .map { and ->
                        TargetingExpression.Disjunction.newBuilder()
                            .addAllOr(
                                and.orList.sortedWith(atomComparator)
                            ).build()
                    }
                    .sortedWith(disjunctionComparator)
            ).build()

    fun toAtom(keyword: KeywordEnum, operation: OperationEnum, value: Any): TargetingExpressionAtom {
        return TargetingExpressionAtom.newBuilder()
            .setKeyword(keyword.number)
            .setOperation(operation.number)
            .setValue(value.toString())
            .build()
    }

    /**
     * Вычисление ContextID по алгоритму БК
     * @implNote портированная версия БКшной реализации на c++
     * @see
     * <a href=https://a.yandex-team.ru/arc/trunk/arcadia/yabs/server/cs/libs/import_operations/ad_groups_show_conditions.cpp?rev=r8647257#L28>
     * NCSImport::NAdGroups::CalcContextID</a>
     */
    fun calcContextID(proto: TargetingExpression): Long {
        if (proto.andList.isNullOrEmpty())
            return 0L

        val messageDigest = MessageDigest.getInstance("MD5")
        proto.andList.forEach { disjunction ->
            disjunction.orList.forEach { atom ->
                val bb = ByteBuffer.allocate(Integer.BYTES * 2).order(ByteOrder.LITTLE_ENDIAN)
                bb.putInt(atom.keyword).putInt(atom.operation)
                messageDigest.update(bb.array())
                messageDigest.update(atom.value.toByteArray())
            }
        }
        val halfMix = getMd5HalfMix(messageDigest.digest())

        val bb = ByteBuffer.allocate(Long.SIZE_BYTES * 2).order(ByteOrder.LITTLE_ENDIAN)
        bb.putLong(0).putLong(halfMix.toLong())
        messageDigest.update(bb.array())
        val halfMix2 = getMd5HalfMix(messageDigest.digest())

        val contextIdMask = BigInteger.valueOf(1).shl(63) - BigInteger.valueOf(1)
        val bigContextIdBit = BigInteger.valueOf(1).shl(62)
        return (halfMix2 and contextIdMask or bigContextIdBit).toLong()
    }
}

