package ru.yandex.direct.logicprocessor.processors.bsexport.feeds;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.springframework.stereotype.Service;

import ru.yandex.adv.direct.feeds.Feed;
import ru.yandex.adv.direct.feeds.FeedAccess;
import ru.yandex.direct.bstransport.yt.repository.feeds.FeedsYtRepository;
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData;
import ru.yandex.direct.common.log.service.LogBsExportEssService;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.entity.feed.repository.FeedRepository;
import ru.yandex.direct.ess.logicobjects.bsexport.feeds.BsExportFeedsObject;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.feed.model.BusinessType.toSource;

@Service
public class BsExportFeedsService {
    private static final String LOG_TYPE = "feed";
    // todo сделать в логе bsexport_data отдельное поле id, для объектов, c ключом не bid, pid или cid
    // тогда отдельный тип для удаленных объектов будет не нужен
    private static final String LOG_DELETED_TYPE = "feed_deleted";
    private final FeedRepository feedRepository;
    private final FeedsYtRepository feedsYtRepository;
    private final LogBsExportEssService logBsExportEssService;

    public BsExportFeedsService(FeedRepository feedRepository, FeedsYtRepository feedsYtRepository,
                                LogBsExportEssService logBsExportEssService) {
        this.feedRepository = feedRepository;
        this.feedsYtRepository = feedsYtRepository;
        this.logBsExportEssService = logBsExportEssService;
    }

    void processFeeds(int shard, List<BsExportFeedsObject> objects) {
        var feedIdsToUpdate = objects.stream()
                .filter(object -> !object.isDeleted())
                .map(BsExportFeedsObject::getFeedId)
                .collect(toList());
        updateFeeds(shard, feedIdsToUpdate);

        var feedIdsToDelete = objects.stream()
                .filter(BsExportFeedsObject::isDeleted)
                .map(BsExportFeedsObject::getFeedId)
                .collect(toList());

        deleteFeeds(feedIdsToDelete);
    }

    private void updateFeeds(int shard, List<Long> feedIds) {
        var feeds = feedRepository.get(shard, feedIds)
                .stream()
                .filter(feed -> feed.getUpdateStatus().equals(UpdateStatus.DONE))
                .collect(toList());

        if (feeds.isEmpty()) {
            return;
        }
        var feedsToModify = feeds.stream()
                .map(
                        feed -> Feed.newBuilder()
                                .setFeedId(feed.getId())
                                // поле в бд not null -> в этом месте NPE быть не может
                                .setBusinessType(toSource(feed.getBusinessType()).getLiteral())
                                .setFeedType(feed.getFeedType().getTypedValue())
                                .setClientId(feed.getClientId())
                                .setIsRemoteUtm(feed.getIsRemoveUtm())
                                .setUrl(feed.getUrl())
                                .build())
                .collect(toList());

        var feedsAccessToModify = feeds.stream()
                .map(
                        feed -> {
                            var feedAccess = FeedAccess.newBuilder()
                                    .setFeedId(feed.getId());
                            feedAccess.setLogin(Objects.nonNull(feed.getLogin()) ? feed.getLogin() : "");
                            feedAccess.setPassword(Objects.nonNull(feed.getPlainPassword()) ?
                                    feed.getPlainPassword() : "");
                            return feedAccess.build();
                        })
                .collect(toList());

        feedsYtRepository.modifyFeeds(feedsToModify);
        feedsYtRepository.modifyFeedsAccess(feedsAccessToModify);
        logFeeds(feedsToModify);
    }

    private void deleteFeeds(List<Long> feedIds) {
        if (feedIds.isEmpty()) {
            return;
        }
        logDeletedFeeds(feedIds);
        feedsYtRepository.delete(feedIds);
    }

    private void logFeeds(List<Feed> feeds) {
        var logsData = feeds.stream()
                .map(feed ->
                        new LogBsExportEssData<Feed>()
                                .withData(feed))
                .collect(Collectors.toList());
        logBsExportEssService.logData(logsData, LOG_TYPE);
    }

    private void logDeletedFeeds(List<Long> feedIds) {
        var logsData = feedIds.stream()
                .map(feedId ->
                        new LogBsExportEssData<Long>()
                                .withData(feedId))
                .collect(Collectors.toList());
        logBsExportEssService.logData(logsData, LOG_DELETED_TYPE);
    }
}
